<?php
// 自动采集页面 - 访问即开始采集，支持URL参数指定类型ID和分类ID
require_once 'config.php';

// 获取数据库连接
$pdo = getDbConnection();

// 初始化变量
$error = '';
$success = '';
$totalProcessed = 0;
$totalSuccess = 0;
$totalFailed = 0;
$totalSkipped = 0; // 记录跳过的重复数据数量

// 获取URL参数
$typeId = isset($_GET['type_id']) ? intval($_GET['type_id']) : 0;
$categoryId = isset($_GET['category_id']) ? intval($_GET['category_id']) : 0;

// 开始自动采集
try {
    // 定义映射关系
    $typeIdMapping = array(
        1 => 1,
        2 => 2,
        3 => 4,
        4 => 3
    );
    
    $categoryIdMapping = array(
        6 => 1,
        7 => 2,
        8 => 3,
        9 => 4,
        10 => 5,
        11 => 6,
        12 => 7,
        20 => 8,
        34 => 9,
        37 => 10,
        13 => 11,
        14 => 12,
        15 => 13,
        16 => 14,
        21 => 15,
        22 => 16,
        23 => 17,
        24 => 18,
        36 => 19,
        29 => 20,
        30 => 21,
        31 => 22,
        32 => 23,
        33 => 24,
        25 => 25,
        26 => 26,
        27 => 27,
        28 => 28
    );
    
    // 设置超时时间
    ini_set('default_socket_timeout', 60);
    
    // 构建采集URL，根据参数添加过滤条件
    $collectUrl = 'http://caiji.dyttzyapi.com/api.php/provide/vod/?ac=detail';
    if ($typeId > 0) {
        $collectUrl .= '&type_id_1=' . $typeId;
    }
    if ($categoryId > 0) {
        $collectUrl .= '&type_id=' . $categoryId;
    }
    
    // 设置HTTP请求头，模拟浏览器访问
    $options = [
        'http' => [
            'header' => "User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36\r\n"
        ]
    ];
    
    $context = stream_context_create($options);
    
    // 获取JSON数据
    $jsonData = @file_get_contents($collectUrl, false, $context);
    
    if ($jsonData === false) {
        // 获取最后一个错误
        $errorInfo = error_get_last();
        throw new Exception('无法连接到采集地址: ' . ($errorInfo ? $errorInfo['message'] : '未知错误'));
    }
    
    // 解析JSON
    $data = json_decode($jsonData, true);
    
    if (json_last_error() !== JSON_ERROR_NONE) {
        throw new Exception('JSON解析失败: ' . json_last_error_msg());
    }
    
    // 检查code是否为1
    if (!isset($data['code']) || $data['code'] != 1) {
        throw new Exception('采集数据格式错误，code不为1');
    }
    
    // 检查是否有list数据
    if (!isset($data['list']) || !is_array($data['list'])) {
        throw new Exception('未找到list数据');
    }
    
    // 重置统计变量
    $totalProcessed = 0;
    $totalSuccess = 0;
    $totalFailed = 0;
    $totalSkipped = 0;
    
    // 开始事务
    $pdo->beginTransaction();
    
    foreach ($data['list'] as $movieData) {
        $totalProcessed++;
        
        try {
            // 检查是否已存在相同的电影
            $existingMovieId = checkExistingMovie($movieData, $pdo, $typeIdMapping, $categoryIdMapping);
            
            if ($existingMovieId) {
                // 电影已存在，增加跳过计数
                $totalSkipped++;
                
                // 更新数据
                $movieId = updateExistingMovie($movieData, $existingMovieId, $pdo, $typeIdMapping, $categoryIdMapping);
                
                // 处理剧集数据 - 总是尝试处理分集数据
                if (isset($movieData['vod_play_url']) && !empty($movieData['vod_play_url'])) {
                    // 尝试处理剧集数据，但不再清空现有剧集
                    $newEpisodesAdded = 0;
                    try {
                        processEpisodeData($movieData, $existingMovieId, $pdo, $newEpisodesAdded);
                    } catch (Exception $e) {
                        // 如果处理分集失败，记录错误但继续处理
                        error_log('处理分集数据失败: ' . $e->getMessage());
                    }
                    
                    // 如果添加了新剧集，则认为更新成功
                    if ($newEpisodesAdded > 0) {
                        $totalSuccess++;
                        // 移除跳过计数，因为更新了新剧集
                        $totalSkipped--;
                    }
                }
                
            } else {
                // 电影不存在，插入新数据
                $movieId = processMovieData($movieData, $pdo, $typeIdMapping, $categoryIdMapping);
                
                // 处理剧集数据 - 总是尝试处理分集数据
                if (isset($movieData['vod_play_url']) && !empty($movieData['vod_play_url'])) {
                    $newEpisodesAdded = 0;
                    // 尝试处理分集数据，但捕获可能的异常以避免整个处理失败
                    try {
                        processEpisodeData($movieData, $movieId, $pdo, $newEpisodesAdded);
                    } catch (Exception $e) {
                        // 如果处理分集失败，记录错误但继续处理
                        error_log('处理分集数据失败: ' . $e->getMessage());
                    }
                }
                
                $totalSuccess++;
            }
        } catch (Exception $e) {
            // 记录错误但继续处理下一个
            error_log('处理电影数据失败: ' . $e->getMessage());
            $totalFailed++;
            continue;
        }
    }
    
    // 提交事务
    $pdo->commit();
    
    $success = "采集完成！总共处理 {$totalProcessed} 个视频，成功 {$totalSuccess} 个，失败 {$totalFailed} 个，跳过 {$totalSkipped} 个重复项";
    
} catch (Exception $e) {
    // 回滚事务
    if ($pdo->inTransaction()) {
        $pdo->rollBack();
    }
    
    $error = '采集失败：' . $e->getMessage();
}

/**
 * 检查是否存在相同的电影
 */
function checkExistingMovie($movieData, $pdo, $typeIdMapping, $categoryIdMapping) {
    // 映射数据
    $type = 1; // 默认类型
    if (isset($movieData['type_id_1']) && isset($typeIdMapping[$movieData['type_id_1']])) {
        $type = $typeIdMapping[$movieData['type_id_1']];
    }
    
    $title = isset($movieData['vod_name']) ? $movieData['vod_name'] : '';
    
    if (empty($title)) {
        throw new Exception('视频标题不能为空');
    }
    
    // 检查数据库中是否已存在相同标题和类型的电影
    $sql = "SELECT id FROM movies WHERE title = ? AND type = ? LIMIT 1";
    $stmt = $pdo->prepare($sql);
    $stmt->execute([$title, $type]);
    $result = $stmt->fetch(PDO::FETCH_ASSOC);
    
    return $result ? $result['id'] : null;
}

/**
 * 计算播放链接中的实际地址数量
 */
function calculateActualEpisodeCount($vodPlayUrl) {
    if (!isset($vodPlayUrl) || empty($vodPlayUrl)) {
        return 1; // 如果没有播放链接，默认为1集
    }
    
    // 提取$$$后面的内容
    $parts = explode('$$$', $vodPlayUrl);
    $urlContent = count($parts) > 1 ? $parts[1] : $vodPlayUrl;
    
    // 使用parseEpisodes函数来获取实际的剧集数量
    $episodes = parseEpisodes($urlContent);
    
    return !empty($episodes) ? count($episodes) : 1;
}

/**
 * 更新已存在的电影数据
 */
function updateExistingMovie($movieData, $movieId, $pdo, $typeIdMapping, $categoryIdMapping) {
    // 映射数据
    $type = 1; // 默认类型
    if (isset($movieData['type_id_1']) && isset($typeIdMapping[$movieData['type_id_1']])) {
        $type = $typeIdMapping[$movieData['type_id_1']];
    }
    
    $categoryId = 1; // 默认分类
    if (isset($movieData['type_id']) && isset($categoryIdMapping[$movieData['type_id']])) {
        $categoryId = $categoryIdMapping[$movieData['type_id']];
    }
    
    // 处理播放链接 - 始终尝试提取第一个可播放URL作为主播放链接
    $playUrl = '';
    $totalEpisodes = isset($movieData['vod_total']) ? intval($movieData['vod_total']) : 1;
    
    // 如果vod_total为0，则计算实际的剧集数量
    if ($totalEpisodes === 0 && isset($movieData['vod_play_url'])) {
        $totalEpisodes = calculateActualEpisodeCount($movieData['vod_play_url']);
    }
    
    // 始终尝试提取第一个可播放URL作为主播放链接
    if (isset($movieData['vod_play_url']) && !empty($movieData['vod_play_url'])) {
        // 提取$$$后面的内容
        $parts = explode('$$$', $movieData['vod_play_url']);
        if (count($parts) > 1) {
            $urlContent = $parts[1];
            
            // 处理链接内容
            $playUrl = extractFirstPlayableUrl($urlContent);
        } else {
            // 如果没有$$$，则直接处理整个链接
            $playUrl = extractFirstPlayableUrl($movieData['vod_play_url']);
        }
    }
    
    // 准备更新数据
    $title = isset($movieData['vod_name']) ? $movieData['vod_name'] : '';
    $subtitle = '';
    if (isset($movieData['vod_sub'])) {
        $subtitle = $movieData['vod_sub'];
    } elseif (isset($movieData['vod_en'])) {
        $subtitle = $movieData['vod_en'];
    }
    
    $director = isset($movieData['vod_director']) ? $movieData['vod_director'] : '';
    $actors = isset($movieData['vod_actor']) ? $movieData['vod_actor'] : '';
    $region = isset($movieData['vod_area']) ? $movieData['vod_area'] : '';
    $language = isset($movieData['vod_lang']) ? $movieData['vod_lang'] : '';
    $releaseYear = isset($movieData['vod_year']) ? $movieData['vod_year'] : '';
    $coverImg = isset($movieData['vod_pic']) ? $movieData['vod_pic'] : '';
    $description = isset($movieData['vod_content']) ? $movieData['vod_content'] : '';
    
    // 更新电影数据
    $sql = "UPDATE movies SET title = ?, subtitle = ?, type = ?, category_id = ?, director = ?, actors = ?, region = ?, language = ?, release_year = ?, cover_img = ?, description = ?, play_url = ?, total_episodes = ? WHERE id = ?";
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute([$title, $subtitle, $type, $categoryId, $director, $actors, $region, $language, $releaseYear, $coverImg, $description, $playUrl, $totalEpisodes, $movieId]);
    
    return $movieId;
}

/**
 * 处理电影数据并保存到数据库
 */
function processMovieData($movieData, $pdo, $typeIdMapping, $categoryIdMapping) {
    // 映射数据
    $type = 1; // 默认类型
    if (isset($movieData['type_id_1']) && isset($typeIdMapping[$movieData['type_id_1']])) {
        $type = $typeIdMapping[$movieData['type_id_1']];
    }
    
    $categoryId = 1; // 默认分类
    if (isset($movieData['type_id']) && isset($categoryIdMapping[$movieData['type_id']])) {
        $categoryId = $categoryIdMapping[$movieData['type_id']];
    }
    
    // 处理播放链接 - 始终尝试提取第一个可播放URL作为主播放链接
    $playUrl = '';
    $totalEpisodes = isset($movieData['vod_total']) ? intval($movieData['vod_total']) : 1;
    
    // 如果vod_total为0，则计算实际的剧集数量
    if ($totalEpisodes === 0 && isset($movieData['vod_play_url'])) {
        $totalEpisodes = calculateActualEpisodeCount($movieData['vod_play_url']);
    }
    
    // 始终尝试提取第一个可播放URL作为主播放链接
    if (isset($movieData['vod_play_url']) && !empty($movieData['vod_play_url'])) {
        // 提取$$$后面的内容
        $parts = explode('$$$', $movieData['vod_play_url']);
        if (count($parts) > 1) {
            $urlContent = $parts[1];
            
            // 处理链接内容
            $playUrl = extractFirstPlayableUrl($urlContent);
        } else {
            // 如果没有$$$，则直接处理整个链接
            $playUrl = extractFirstPlayableUrl($movieData['vod_play_url']);
        }
    }
    
    // 准备插入数据
    $title = isset($movieData['vod_name']) ? $movieData['vod_name'] : '';
    $subtitle = '';
    if (isset($movieData['vod_sub'])) {
        $subtitle = $movieData['vod_sub'];
    } elseif (isset($movieData['vod_en'])) {
        $subtitle = $movieData['vod_en'];
    }
    
    $director = isset($movieData['vod_director']) ? $movieData['vod_director'] : '';
    $actors = isset($movieData['vod_actor']) ? $movieData['vod_actor'] : '';
    $region = isset($movieData['vod_area']) ? $movieData['vod_area'] : '';
    $language = isset($movieData['vod_lang']) ? $movieData['vod_lang'] : '';
    $releaseYear = isset($movieData['vod_year']) ? $movieData['vod_year'] : '';
    $coverImg = isset($movieData['vod_pic']) ? $movieData['vod_pic'] : '';
    $description = isset($movieData['vod_content']) ? $movieData['vod_content'] : '';
    
    // 验证必填字段
    if (empty($title)) {
        throw new Exception('视频标题不能为空');
    }
    
    // 插入电影数据
    $sql = "INSERT INTO movies (title, subtitle, type, category_id, director, actors, region, language, release_year, cover_img, description, play_url, total_episodes, status, create_time) 
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())";
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute([$title, $subtitle, $type, $categoryId, $director, $actors, $region, $language, $releaseYear, $coverImg, $description, $playUrl, $totalEpisodes, 1]);
    
    // 返回插入的电影ID
    return $pdo->lastInsertId();
}

/**
 * 处理剧集数据并保存到数据库
 */
function processEpisodeData($movieData, $movieId, $pdo, &$newEpisodesAdded) {
    $newEpisodesAdded = 0; // 初始化新增剧集计数
    
    if (!isset($movieData['vod_play_url']) || empty($movieData['vod_play_url'])) {
        throw new Exception('vod_play_url不存在或为空');
    }
    
    // 提取$$$后面的内容
    $parts = explode('$$$', $movieData['vod_play_url']);
    $urlContent = count($parts) > 1 ? $parts[1] : $movieData['vod_play_url'];
    
    // 解析剧集数据
    $episodes = parseEpisodes($urlContent);
    
    if (empty($episodes)) {
        throw new Exception('未找到有效的剧集数据');
    }
    
    // 插入分集数据
    foreach ($episodes as $episodeNum => $episodeData) {
        // 跳过空链接
        if (empty(trim($episodeData['url']))) {
            continue;
        }
        
        // 替换m3u8链接
        $playUrl = str_replace('index.m3u8', '3000k/hls/mixed.m3u8', $episodeData['url']);
        
        // 检查是否已存在相同的剧集
        if (checkExistingEpisode($movieId, $episodeNum, $pdo)) {
            continue; // 跳过已存在的剧集，但不计入跳过计数
        }
        
        // 准备SQL语句
        if (!empty($episodeData['title'])) {
            $sql = "INSERT INTO episodes (movie_id, episode_num, episode_title, play_url, create_time) VALUES (?, ?, ?, ?, NOW())";
            $stmt = $pdo->prepare($sql);
            $stmt->execute([$movieId, $episodeNum, $episodeData['title'], $playUrl]);
        } else {
            $sql = "INSERT INTO episodes (movie_id, episode_num, play_url, create_time) VALUES (?, ?, ?, NOW())";
            $stmt = $pdo->prepare($sql);
            $stmt->execute([$movieId, $episodeNum, $playUrl]);
        }
        
        $newEpisodesAdded++; // 增加新增剧集计数
    }
}

/**
 * 检查是否存在相同的剧集
 */
function checkExistingEpisode($movieId, $episodeNum, $pdo) {
    $sql = "SELECT id FROM episodes WHERE movie_id = ? AND episode_num = ? LIMIT 1";
    $stmt = $pdo->prepare($sql);
    $stmt->execute([$movieId, $episodeNum]);
    $result = $stmt->fetch(PDO::FETCH_ASSOC);
    
    return $result !== false;
}

/**
 * 解析剧集数据，支持#和$分隔符
 */
function parseEpisodes($urlContent) {
    $episodes = array();
    $episodeNum = 1;
    
    // 首先尝试用#分割（主要分隔符）
    $segments = explode('#', $urlContent);
    
    if (count($segments) > 1) {
        foreach ($segments as $segment) {
            if (empty(trim($segment))) {
                continue;
            }
            
            // 检查是否有$分隔的标题和链接
            if (strpos($segment, '$') !== false) {
                list($title, $url) = explode('$', $segment, 2);
                $episodes[$episodeNum++] = array(
                    'title' => trim($title),
                    'url' => trim($url)
                );
            } else {
                // 可能是单独的标题或链接
                $episodes[$episodeNum++] = array(
                    'title' => '',
                    'url' => trim($segment)
                );
            }
        }
    } else {
        // 如果#分割失败，尝试用$分割
        $segments = explode('$', $urlContent);
        
        for ($i = 0; $i < count($segments); $i += 2) {
            if (isset($segments[$i+1]) && !empty(trim($segments[$i+1]))) {
                $episodes[$episodeNum++] = array(
                    'title' => trim($segments[$i]),
                    'url' => trim($segments[$i+1])
                );
            }
        }
    }
    
    return $episodes;
}

/**
 * 提取第一个可播放的URL
 */
function extractFirstPlayableUrl($urlContent) {
    // 解析所有剧集
    $episodes = parseEpisodes($urlContent);
    
    // 返回第一个有效的URL
    foreach ($episodes as $episode) {
        if (!empty($episode['url'])) {
            // 替换m3u8链接
            return str_replace('index.m3u8', '3000k/hls/mixed.m3u8', $episode['url']);
        }
    }
    
    return '';
}

// 显示采集结果
?>
<!DOCTYPE html>
<html lang="zh-CN">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>自动视频采集 - 爱影视</title>
    <link rel="stylesheet" href="css/style.css">
</head>
<body>
    <div class="container mx-auto px-4 py-8 max-w-4xl">
        <h1 class="text-2xl font-bold mb-6 text-center">视频自动采集</h1>
        
        <?php if (!empty($error)): ?>
            <div class="error-message bg-red-500 text-white p-4 rounded-md mb-4">
                <?php echo $error; ?>
            </div>
        <?php endif; ?>
        
        <?php if (!empty($success)): ?>
            <div class="success-message bg-green-500 text-white p-4 rounded-md mb-4">
                <?php echo $success; ?>
            </div>
        <?php endif; ?>
        
        <div class="info-box bg-gray-800 p-4 rounded-md mb-4">
            <h3 class="text-lg font-semibold mb-2">采集参数</h3>
            <p>类型ID: <?php echo $typeId > 0 ? $typeId : '未指定'; ?></p>
            <p>分类ID: <?php echo $categoryId > 0 ? $categoryId : '未指定'; ?></p>
        </div>
        
        <div class="usage-guide bg-gray-800 p-4 rounded-md">
            <h3 class="text-lg font-semibold mb-2">使用指南</h3>
            <ul class="list-disc pl-5 space-y-1">
                <li>直接访问此页面将采集所有类型的视频</li>
                <li>添加URL参数可指定采集范围，例如：</li>
                <li>- 仅采集特定类型：<code>auto_collect.php?type_id=1</code></li>
                <li>- 仅采集特定分类：<code>auto_collect.php?category_id=6</code></li>
                <li>- 同时指定类型和分类：<code>auto_collect.php?type_id=1&category_id=6</code></li>
            </ul>
        </div>
    </div>
</body>
</html>