<?php
// B站视频采集功能页面
require_once 'includes/check_login.php';
require_once '../config.php';

// 数据库连接
$pdo = getDbConnection();

// 初始化变量
$error = '';
$success = '';

// 获取视频类型列表
$videoTypes = $pdo->query("SELECT id, name FROM video_types WHERE status = 1 ORDER BY sort_order ASC")->fetchAll();

// 获取所有分类列表并按类型分组
$categories = array();
$categoryResult = $pdo->query("SELECT id, name, type FROM movie_categories WHERE status = 1 ORDER BY type, id ASC");
while ($cat = $categoryResult->fetch(PDO::FETCH_ASSOC)) {
    if (!isset($categories[$cat['type']])) {
        $categories[$cat['type']] = array();
    }
    $categories[$cat['type']][] = $cat;
}

// 处理表单提交
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['collect_bilibili'])) {
    try {
        // 获取表单数据
        $videoUrl = trim($_POST['video_url']);
        $selectedType = intval($_POST['video_type']);
        $selectedCategory = intval($_POST['video_category']);
        
        // 验证必填字段
        if (empty($videoUrl)) {
            throw new Exception('请输入B站视频链接');
        }
        
        if ($selectedType <= 0) {
            throw new Exception('请选择视频类型');
        }
        
        if ($selectedCategory <= 0) {
            throw new Exception('请选择视频分类');
        }
        
        // 从链接中提取BVID - 更宽松的正则表达式
        preg_match('/(BV\w+)/i', $videoUrl, $matches);
        if (empty($matches[1])) {
            // 如果正则匹配失败，尝试直接从URL中提取
            $urlParts = parse_url($videoUrl);
            if (isset($urlParts['query'])) {
                parse_str($urlParts['query'], $queryParams);
                if (isset($queryParams['bvid'])) {
                    $bvid = $queryParams['bvid'];
                } else if (isset($queryParams['aid'])) {
                    // 如果提供了aid，需要转换为bvid
                    throw new Exception('暂不支持aid格式，请提供包含BVID的链接');
                }
            }
            if (empty($bvid)) {
                throw new Exception('无法从链接中提取BVID，请检查链接格式是否正确');
            }
        } else {
            $bvid = $matches[1];
        }
        
        // 调试信息
        $debugInfo = "BVID: {$bvid}\n";
        
        // 使用cURL替代file_get_contents，更稳定
        $apiUrl = "https://api.bilibili.com/x/web-interface/view?bvid={$bvid}";
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $apiUrl);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 15); // 15秒超时
        curl_setopt($ch, CURLOPT_HTTPHEADER, array(
            'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36',
            'Referer: https://www.bilibili.com/',
            'Accept: application/json, text/javascript, */*; q=0.01',
            'Accept-Language: zh-CN,zh;q=0.9',
            'Connection: keep-alive'
        ));
        
        $jsonData = curl_exec($ch);
        $curlErrno = curl_errno($ch);
        $curlError = curl_error($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        
        curl_close($ch);
        
        // 添加调试信息
        $debugInfo .= "HTTP状态码: {$httpCode}\n";
        if ($curlErrno) {
            $debugInfo .= "cURL错误: {$curlError}\n";
            throw new Exception("API请求失败: {$curlError}");
        }
        
        if (empty($jsonData)) {
            $debugInfo .= "API返回空数据\n";
            throw new Exception('B站API返回空数据，请检查链接是否有效或稍后再试');
        }
        
        // 解析JSON数据
        $data = json_decode($jsonData, true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            $debugInfo .= "JSON解析错误: " . json_last_error_msg() . "\n";
            $debugInfo .= "原始数据: " . substr($jsonData, 0, 200) . "...\n";
            throw new Exception('解析B站API返回数据失败: ' . json_last_error_msg());
        }
        
        // 检查API返回状态
        $debugInfo .= "API返回状态码: " . (isset($data['code']) ? $data['code'] : '未知') . "\n";
        if (isset($data['message'])) {
            $debugInfo .= "API返回消息: {$data['message']}\n";
        }
        
        if (!isset($data['code']) || $data['code'] != 0 || !isset($data['data'])) {
            $errorMsg = 'B站API返回数据格式错误或获取失败';
            if (isset($data['message'])) {
                $errorMsg .= ': ' . $data['message'];
            }
            throw new Exception($errorMsg);
        }
        
        $videoData = $data['data'];
        
        // 开始事务
        $pdo->beginTransaction();
        
        try {
            // 检查电影是否已存在
            $stmt = $pdo->prepare("SELECT id FROM movies WHERE title = :title LIMIT 1");
            $stmt->execute(['title' => $videoData['title']]);
            $existingMovie = $stmt->fetch();
            
            if ($existingMovie) {
                $movieId = $existingMovie['id'];
                $success .= "视频《{$videoData['title']}》已存在，使用现有ID: {$movieId}\n";
            } else {
                $stmt = $pdo->prepare("INSERT INTO movies (title, subtitle, actors, region, language, description, cover_img, director, release_year, type, category_id, status, total_episodes, play_url) VALUES (:title, :subtitle, :actors, :region, :language, :description, :cover_img, :director, :release_year, :type, :category_id, 1, :total_episodes, :play_url)");
                $stmt->execute([
                    'title' => $videoData['title'],
                    'subtitle' => '',
                    'actors' => '',
                    'region' => '',
                    'language' => '',
                    'description' => '<p>'.$videoData['desc'] ?? ''.'</p>',
                    'cover_img' => $videoData['pic'] ?? '',
                    'director' => $videoData['owner']['name'] ?? '未知',
                    'release_year' => $videoData['tid_v2'] ?? 0,
                    'type' => $selectedType,
                    'category_id' => $selectedCategory,
                    'total_episodes' => isset($videoData['pages']) ? count($videoData['pages']) : 1,
                    'play_url' => isset($videoData['pages'][0]) ? "bilibili/index.php?url={$bvid}&p={$videoData['pages'][0]['page']}" : "bilibili/index.php?url={$bvid}" // 修改为使用BVID
                ]);
                $movieId = $pdo->lastInsertId();
                $success .= "成功添加视频《{$videoData['title']}》\n";
            }
            
            // 处理剧集数据
            if (isset($videoData['pages']) && is_array($videoData['pages'])) {
                $episodesAdded = 0;
                
                foreach ($videoData['pages'] as $page) {
                    // 构建播放链接 - 修改为使用BVID
                    $playUrl = "bilibili/index.php?url={$bvid}&p={$page['page']}";
                    
                    // 检查剧集是否已存在
                    $stmt = $pdo->prepare("SELECT id FROM episodes WHERE movie_id = :movie_id AND episode_num = :episode_num LIMIT 1");
                    $stmt->execute([
                        'movie_id' => $movieId,
                        'episode_num' => $page['page']
                    ]);
                    $existingEpisode = $stmt->fetch();
                    
                    if ($existingEpisode) {
                        // 更新现有剧集 - 修复表结构不匹配的问题
                        $stmt = $pdo->prepare("UPDATE episodes SET episode_title = :episode_title, play_url = :play_url WHERE id = :id");
                        $stmt->execute([
                            'episode_title' => $page['part'] ?? "第{$page['page']}集",
                            'play_url' => $playUrl,
                            'id' => $existingEpisode['id']
                        ]);
                    } else {
                        // 插入新剧集 - 修复表结构不匹配的问题（移除不存在的status字段）
                        $stmt = $pdo->prepare("INSERT INTO episodes (movie_id, episode_num, episode_title, play_url) VALUES (:movie_id, :episode_num, :episode_title, :play_url)");
                        $stmt->execute([
                            'movie_id' => $movieId,
                            'episode_num' => $page['page'] ?? 1,
                            'episode_title' => $page['part'] ?? "第{$page['page']}集",
                            'play_url' => $playUrl
                        ]);
                        $episodesAdded++;
                    }
                }
                
                if ($episodesAdded > 0) {
                    $success .= "成功添加 {$episodesAdded} 个剧集\n";
                } else {
                    $success .= "所有剧集已存在，无需重复添加\n";
                }
            } else {
                throw new Exception('未找到剧集信息');
            }
            
            // 提交事务
            $pdo->commit();
            
        } catch (Exception $e) {
            // 回滚事务
            $pdo->rollBack();
            throw new Exception('数据库操作失败: ' . $e->getMessage());
        }
        
    } catch (Exception $e) {
        $error = $e->getMessage();
        // 如果有调试信息，添加到错误信息中
        if (!empty($debugInfo)) {
            $error .= "\n\n调试信息:\n" . $debugInfo;
        }
    }
}
?>

<!DOCTYPE html>
<html lang="zh-CN">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>B站视频采集 - 后台管理</title>
    <link rel="stylesheet" href="css/style.css">
    <script src="https://registry.npmmirror.com/axios/1.12.2/files/dist/axios.min.js"></script>
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            // 类型选择变化时，更新分类列表
            const videoTypeSelect = document.getElementById('video_type');
            const videoCategorySelect = document.getElementById('video_category');
            
            function updateCategoryOptions() {
                const selectedType = videoTypeSelect.value;
                const allOptions = videoCategorySelect.querySelectorAll('option[data-type]');
                
                // 隐藏所有选项
                allOptions.forEach(option => {
                    option.style.display = 'none';
                });
                
                // 显示选中类型的选项
                const selectedOptions = videoCategorySelect.querySelectorAll(`option[data-type="${selectedType}"]`);
                selectedOptions.forEach(option => {
                    option.style.display = 'block';
                });
                
                // 重置选中状态
                videoCategorySelect.value = '';
            }
            
            // 初始更新
            updateCategoryOptions();
            
            // 添加事件监听
            videoTypeSelect.addEventListener('change', updateCategoryOptions);
        });
    </script>
</head>
<body>
    <?php include 'includes/header.php'; ?>
    
    <div class="container">
        <div class="content-wrapper">
            <?php include 'includes/sidebar.php'; ?>
            
            <div class="content">
                <div class="main-content">
                    <div class="breadcrumb">
                        <a href="index.php">首页</a> &gt; 
                        <span>B站视频采集</span>
                    </div>
                    
                    <div class="panel">
                        <div class="panel-header">
                            <h3>B站视频采集</h3>
                        </div>
                        <div class="panel-body">
                            
                            <?php if ($error): ?>
                                <div class="alert alert-error">
                                    <?php echo $error; ?>
                                </div>
                            <?php endif; ?>
                            
                            <?php if ($success): ?>
                                <div class="alert alert-success">
                                    <?php echo nl2br($success); ?>
                                </div>
                            <?php endif; ?>
                            
                            <form method="post" action="bilibili_collect.php">
                                <div class="form-group">
                                    <label for="video_url">B站视频链接</label>
                                    <input type="text" id="video_url" name="video_url" class="form-control" placeholder="请输入B站视频链接，例如：https://www.bilibili.com/video/BV1Qq421c7sQ" value="<?php echo isset($_POST['video_url']) ? htmlspecialchars($_POST['video_url']) : ''; ?>">
                                    <small class="form-text text-muted">请确保链接格式为包含/video/BVxxxxx的完整B站视频链接</small>
                                </div>
                                
                                <div class="form-group">
                                    <label for="video_type">视频类型</label>
                                    <select id="video_type" name="video_type" class="form-control">
                                        <option value="">请选择视频类型</option>
                                        <?php foreach ($videoTypes as $type): ?>
                                            <option value="<?php echo $type['id']; ?>" <?php echo isset($_POST['video_type']) && intval($_POST['video_type']) === $type['id'] ? 'selected' : ''; ?>><?php echo $type['name']; ?></option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                                
                                <div class="form-group">
                                    <label for="video_category">视频分类</label>
                                    <select id="video_category" name="video_category" class="form-control">
                                        <option value="">请选择视频分类</option>
                                        <?php foreach ($categories as $typeId => $catList): ?>
                                            <?php foreach ($catList as $category): ?>
                                                <option value="<?php echo $category['id']; ?>" data-type="<?php echo $category['type']; ?>" <?php echo isset($_POST['video_category']) && intval($_POST['video_category']) === $category['id'] ? 'selected' : ''; ?>><?php echo $category['name']; ?></option>
                                            <?php endforeach; ?>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                                
                                <button type="submit" name="collect_bilibili" class="btn btn-primary">开始采集</button>
                            </form>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <?php include 'includes/footer.php'; ?>
</body>
</html>