<?php
// 分类管理页面
require_once 'includes/check_login.php';
require_once '../config.php';

// 添加 session 初始化
session_start();

// 在文件顶部，数据库连接代码后添加处理删除操作的代码
// 获取数据库连接
$pdo = getDbConnection();

// 处理删除操作
$error = '';
$success = '';
if (isset($_GET['action']) && $_GET['action'] == 'delete' && isset($_GET['id'])) {
    $id = intval($_GET['id']);
    
    // 检查是否有视频正在使用此分类
    $stmt = $pdo->prepare("SELECT COUNT(*) FROM movies WHERE category_id = ?");
    $stmt->execute([$id]);
    $count = $stmt->fetchColumn();
    
    if ($count > 0) {
        $error = '该分类下有视频数据，无法删除！请先修改这些视频的分类。';
    } else {
        // 删除分类
        $stmt = $pdo->prepare("DELETE FROM movie_categories WHERE id = ?");
        if ($stmt->execute([$id])) {
            // 使用 session 存储成功消息
            $_SESSION['success_message'] = '分类删除成功';
            // 重定向到不带任何参数的页面
            header("Location: category_list.php");
            exit;
        } else {
            $error = '分类删除失败，请重试';
        }
    }
}

// 添加批量操作处理逻辑
session_start();

if (isset($_POST['action'])) {
    try {
        // 开始事务
        $pdo->beginTransaction();
        
        if ($_POST['action'] == 'batch_delete' && isset($_POST['category_ids'])) {
            $categoryIds = array_map('intval', $_POST['category_ids']);
            
            if (empty($categoryIds)) {
                throw new Exception('请选择要删除的分类');
            }
            
            // 检查是否有视频正在使用这些分类
            $placeholders = implode(',', array_fill(0, count($categoryIds), '?'));
            $stmt = $pdo->prepare("SELECT COUNT(*) FROM movies WHERE category_id IN ($placeholders)");
            $stmt->execute($categoryIds);
            $count = $stmt->fetchColumn();
            
            if ($count > 0) {
                throw new Exception('选中的部分分类下有视频数据，无法删除！请先修改这些视频的分类。');
            }
            
            // 批量删除分类
            $stmt = $pdo->prepare("DELETE FROM movie_categories WHERE id IN ($placeholders)");
            $stmt->execute($categoryIds);
            
            // 提交事务
            $pdo->commit();
            
            // 使用 session 存储成功消息
            $_SESSION['success_message'] = '批量删除成功';
        } elseif ($_POST['action'] == 'batch_change_status' && isset($_POST['category_ids']) && isset($_POST['status'])) {
            $categoryIds = array_map('intval', $_POST['category_ids']);
            $status = intval($_POST['status']);
            
            if (empty($categoryIds)) {
                throw new Exception('请选择要更改状态的分类');
            }
            
            // 批量更改状态
            $placeholders = implode(',', array_fill(0, count($categoryIds), '?'));
            // 修改为统一使用位置参数
            $stmt = $pdo->prepare("UPDATE movie_categories SET status = ? WHERE id IN ($placeholders)");
            
            // 准备参数数组，第一个参数是状态值，后面跟着所有分类ID
            $params = array_merge([$status], $categoryIds);
            $stmt->execute($params);
            
            // 提交事务
            $pdo->commit();
            
            // 使用 session 存储成功消息
            $_SESSION['success_message'] = '批量更改状态成功';
        }
        
        // 重定向到不带任何参数的页面
        header("Location: category_list.php");
        exit;
    } catch (Exception $e) {
        // 回滚事务
        $pdo->rollBack();
        $error = $e->getMessage();
    }
}

// 检查 session 中的成功消息
if (isset($_SESSION['success_message'])) {
    $success = $_SESSION['success_message'];
    // 清除 session 中的消息，防止页面刷新后再次显示
    unset($_SESSION['success_message']);
}

// 获取视频类型列表
$videoTypes = array();
$sqlTypes = "SELECT id, name FROM video_types WHERE status = 1 ORDER BY sort_order ASC";
$stmtTypes = $pdo->prepare($sqlTypes);
$stmtTypes->execute();
while ($row = $stmtTypes->fetch(PDO::FETCH_ASSOC)) {
    $videoTypes[$row['id']] = $row['name'] . '分类';
}

// 获取当前选择的类型筛选条件
$selectedType = isset($_GET['type']) ? intval($_GET['type']) : 0;

// 添加分页参数处理
$page = isset($_GET['page']) ? intval($_GET['page']) : 1;
// 确保页码至少为1
$page = max(1, $page);
$pageSize = 20;
$offset = ($page - 1) * $pageSize;

// 构建SQL查询 - 包含类型筛选条件
// 修改前的查询
// $sql = "SELECT * FROM movie_categories";
// 修改后的查询，添加LEFT JOIN和COUNT统计
$sql = "SELECT c.*, COUNT(m.id) AS video_count FROM movie_categories c
        LEFT JOIN movies m ON c.id = m.category_id";
if ($selectedType > 0) {
    $sql .= " WHERE c.type = :type";
}

// 获取总记录数
// 修复前: sqlCount = str_replace(...)\n// 修复后:
$sqlCount = str_replace('c.*, COUNT(m.id) AS video_count', 'c.id, COUNT(m.id) AS video_count', $sql);
$countStmt = $pdo->prepare($sqlCount . " GROUP BY c.id");
if ($selectedType > 0) {
    $countStmt->bindParam(':type', $selectedType, PDO::PARAM_INT);
}
$countStmt->execute();
$totalCount = $countStmt->rowCount();
$totalPages = ceil($totalCount / $pageSize);

// 添加排序和分页
// 修复前: sql .= " GROUP BY c.id ORDER BY c.id ASC LIMIT :offset, :pageSize";\n// 修复后:
$sql .= " GROUP BY c.id ORDER BY c.id ASC LIMIT :offset, :pageSize";

$stmt = $pdo->prepare($sql);
if ($selectedType > 0) {
    $stmt->bindParam(':type', $selectedType, PDO::PARAM_INT);
}
$stmt->bindParam(':offset', $offset, PDO::PARAM_INT);
$stmt->bindParam(':pageSize', $pageSize, PDO::PARAM_INT);
$stmt->execute();
$categories = $stmt->fetchAll();
?>
<!DOCTYPE html>
<html lang="zh-CN">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>分类管理 - 爱影视</title>
    <link rel="stylesheet" href="css/style.css">
    <script src="js/main.js"></script>
</head>
<body>
    <?php include 'includes/header.php'; ?>
    
    <div class="container">
        <div class="content-wrapper">
            <?php include 'includes/sidebar.php'; ?>
            
            <div class="content">
                <div class="main-content">
                    <div class="page-header">
                        <h2>分类管理</h2>
                        <p>注意！删除分类前先确认是否有视频数据，否则无法删除。</p>
                        <div class="header-actions">
                            <div class="filter-form">
                                <form method="get" action="category_list.php">
                                    <select name="type" onchange="this.form.submit()">
                                        <option value="0">全部类型</option>
                                        <?php foreach ($videoTypes as $typeId => $typeName): ?>
                                            <option value="<?php echo $typeId; ?>" <?php echo ($selectedType == $typeId) ? 'selected' : ''; ?>><?php echo $typeName; ?></option>
                                        <?php endforeach; ?>
                                    </select>
                                </form>
                            </div>
                            <a href="category_edit.php" class="btn-primary">添加分类</a>
                        </div>
                    </div>
                
                    <?php if (!empty($error)): ?>
                        <div class="alert alert-error"><?php echo $error; ?></div>
                    <?php endif; ?>
                
                    <?php if (!empty($success)): ?>
                        <div class="alert alert-success"><?php echo $success; ?></div>
                    <?php endif; ?>
                    
                    <!-- 添加批量操作表单 -->
                    <form id="batch-form" method="post" action="category_list.php">
                        <input type="hidden" name="action" id="batch-action">
                        <input type="hidden" name="status" id="batch-status-value">
                        <input type="hidden" name="type" value="<?php echo $selectedType; ?>">
                        <input type="hidden" name="page" value="<?php echo $page; ?>">
                        
                        <!-- 现有的表格代码 -->
                        <table class="data-table">
                            <thead>
                                <tr>
                                    <th><input type="checkbox" id="select-all"></th>
                                    <th>ID</th>
                                    <th>分类名称</th>
                                    <th>类型</th>
                                    <th>视频数量</th> <!-- 添加视频数量列头 -->
                                    <th>状态</th>
                                    <th>操作</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if (empty($categories)): ?>
                                    <tr>
                                        <td colspan="7" class="no-data">暂无分类数据</td> <!-- 更新colspan为7 -->
                                    </tr>
                                <?php else: ?>
                                    <?php foreach ($categories as $category): ?>
                                        <tr>
                                            <td><input type="checkbox" name="category_ids[]" class="category-checkbox" value="<?php echo $category['id']; ?>"></td>
                                            <td><?php echo $category['id']; ?></td>
                                            <td><?php echo $category['name']; ?></td>
                                            <td>
                                                <?php 
                                                echo isset($videoTypes[$category['type']]) ? $videoTypes[$category['type']] : '未知类型';
                                                ?>
                                            </td>
                                            <td><?php echo $category['video_count']; ?></td> <!-- 显示视频数量 -->
                                            <td><?php echo $category['status'] == 1 ? '<span class="status-active">启用</span>' : '<span class="status-inactive">禁用</span>'; ?></td>
                                            <td>
                                                <a href="category_edit.php?id=<?php echo $category['id']; ?>" class="btn-edit">编辑</a>
                                                <a href="category_list.php?action=delete&id=<?php echo $category['id']; ?>" class="btn-delete" onclick="return confirm('确定要删除该分类吗？删除后关联的视频将无法正常显示！');">删除</a>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </tbody>
                        </table>
                        
                        <!-- 批量操作容器 -->
                        <?php if (!empty($categories)): ?>
                        <div class="batch-actions-container">
                            <div class="batch-actions">
                                <div class="batch-info">
                                    <label><input type="checkbox" id="select-all-bottom"> 全选</label>
                                    <span>已选择 <span id="selected-count">0</span> 项</span>
                                </div>
                                <div class="batch-controls">
                                    <select id="batch-status-select">
                                        <option value="">-- 更改状态 --</option>
                                        <option value="1">设为启用</option>
                                        <option value="0">设为禁用</option>
                                    </select>
                                    <button type="button" id="batch-status-btn" class="btn-edit">批量更改状态</button>
                                    <button type="button" id="batch-delete-btn" class="btn-delete">批量删除</button>
                                </div>
                            </div>
                        </div>
                        <script>
                            // 获取两个全选框元素
                            const selectAllTop = document.getElementById('select-all');
                            const selectAllBottom = document.getElementById('select-all-bottom');
                            
                            // 获取所有分类复选框
                            const categoryCheckboxes = document.querySelectorAll('.category-checkbox');
                            
                            // 两个全选框状态同步
                            selectAllTop.addEventListener('change', function() {
                                selectAllBottom.checked = this.checked;
                                updateAllCheckboxes(this.checked);
                                updateSelectedCount();
                            });
                            
                            selectAllBottom.addEventListener('change', function() {
                                selectAllTop.checked = this.checked;
                                updateAllCheckboxes(this.checked);
                                updateSelectedCount();
                            });
                            
                            // 更新所有分类复选框的状态
                            function updateAllCheckboxes(checked) {
                                categoryCheckboxes.forEach(checkbox => {
                                    checkbox.checked = checked;
                                });
                            }
                            
                            // 为每个分类复选框添加事件监听
                            categoryCheckboxes.forEach(checkbox => {
                                checkbox.addEventListener('change', function() {
                                    updateSelectAllStatus();
                                    updateSelectedCount();
                                });
                            });
                            
                            // 更新全选框的状态
                            function updateSelectAllStatus() {
                                const totalCheckboxes = categoryCheckboxes.length;
                                const checkedCheckboxes = document.querySelectorAll('.category-checkbox:checked').length;
                                
                                selectAllTop.checked = checkedCheckboxes === totalCheckboxes;
                                selectAllBottom.checked = checkedCheckboxes === totalCheckboxes;
                                
                                // 设置indeterminate状态
                                selectAllTop.indeterminate = checkedCheckboxes > 0 && checkedCheckboxes < totalCheckboxes;
                                selectAllBottom.indeterminate = checkedCheckboxes > 0 && checkedCheckboxes < totalCheckboxes;
                            }
                            
                            // 更新选中数量显示
                            function updateSelectedCount() {
                                const selectedCount = document.querySelectorAll('.category-checkbox:checked').length;
                                document.getElementById('selected-count').textContent = selectedCount;
                            }
                            
                            // 批量更改状态
                            document.getElementById('batch-status-btn').addEventListener('click', function() {
                                const selectedStatus = document.getElementById('batch-status-select').value;
                                const selectedCategories = getSelectedCategories();
                                
                                if (selectedCategories.length === 0) {
                                    alert('请至少选择一个分类');
                                    return;
                                }
                                
                                if (!selectedStatus) {
                                    alert('请选择要更改的状态');
                                    return;
                                }
                                
                                document.getElementById('batch-action').value = 'batch_change_status';
                                document.getElementById('batch-status-value').value = selectedStatus;
                                document.getElementById('batch-form').submit();
                            });
                            
                            // 批量删除
                            document.getElementById('batch-delete-btn').addEventListener('click', function() {
                                const selectedCategories = getSelectedCategories();
                                
                                if (selectedCategories.length === 0) {
                                    alert('请至少选择一个分类');
                                    return;
                                }
                                
                                if (confirm('确定要删除选中的' + selectedCategories.length + '个分类吗？如果分类下有视频数据，将无法删除。')) {
                                    document.getElementById('batch-action').value = 'batch_delete';
                                    document.getElementById('batch-form').submit();
                                }
                            });
                            
                            // 获取选中的分类ID
                            function getSelectedCategories() {
                                const checkboxes = document.querySelectorAll('.category-checkbox:checked');
                                const categoryIds = Array.from(checkboxes).map(checkbox => checkbox.value);
                                return categoryIds;
                            }
                            
                            // 初始化选中数量显示
                            window.onload = function() {
                                updateSelectedCount();
                            };
                        </script>
                        <?php endif; ?>
                    </form>
                    
                    <!-- 分页控件 -->
                    <?php if ($totalPages > 1): ?>
                        <div class="pagination">
                            <a href="category_list.php?type=<?php echo $selectedType; ?>&page=1" <?php echo $page == 1 ? 'class="disabled"' : ''; ?>>首页</a>
                            <a href="category_list.php?type=<?php echo $selectedType; ?>&page=<?php echo $page - 1; ?>" <?php echo $page == 1 ? 'class="disabled"' : ''; ?>>上一页</a>
                            
                            <?php for ($i = max(1, $page - 2); $i <= min($totalPages, $page + 2); $i++): ?>
                                <a href="category_list.php?type=<?php echo $selectedType; ?>&page=<?php echo $i; ?>" <?php echo $page == $i ? 'class="active"' : ''; ?>><?php echo $i; ?></a>
                            <?php endfor; ?>
                            
                            <a href="category_list.php?type=<?php echo $selectedType; ?>&page=<?php echo $page + 1; ?>" <?php echo $page == $totalPages ? 'class="disabled"' : ''; ?>>下一页</a>
                            <a href="category_list.php?type=<?php echo $selectedType; ?>&page=<?php echo $totalPages; ?>" <?php echo $page == $totalPages ? 'class="disabled"' : ''; ?>>末页</a>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
    
    <?php include 'includes/footer.php'; ?>
</body>
</html>