<?php
// 留言管理页面
require_once 'includes/check_login.php';
require_once '../config.php';

// 初始化session，用于存储操作结果消息
session_start();

// 在文件顶部，数据库连接代码后添加处理删除操作的代码
// 获取数据库连接
$pdo = getDbConnection();

// 处理删除操作
$error = '';
$success = '';
// 检查session中的成功消息
if (isset($_SESSION['success_message']) && !empty($_SESSION['success_message'])) {
    $success = $_SESSION['success_message'];
    // 显示后清除session中的消息
    unset($_SESSION['success_message']);
}

// 处理单个删除操作
if (isset($_GET['action']) && $_GET['action'] == 'delete' && isset($_GET['id'])) {
    $comment_id = intval($_GET['id']);
    
    try {
        // 执行删除操作
        $sql = "DELETE FROM comments WHERE id = ?";
        $stmt = $pdo->prepare($sql);
        if ($stmt->execute([$comment_id])) {
            $success = '留言删除成功';
        } else {
            $error = "删除失败";
        }
    } catch (PDOException $e) {
        $error = "数据库错误：" . $e->getMessage();
    }
}

// 处理批量删除操作
if (isset($_POST['action']) && $_POST['action'] == 'batch_delete') {
    if (isset($_POST['comment_ids']) && !empty($_POST['comment_ids'])) {
        $comment_ids = $_POST['comment_ids'];
        $current_status = isset($_POST['status']) ? intval($_POST['status']) : -1;
        $current_search = isset($_POST['search']) ? $_POST['search'] : '';
        
        try {
            // 开启事务
            $pdo->beginTransaction();
            
            // 批量删除留言
            $placeholders = implode(',', array_fill(0, count($comment_ids), '?'));
            $sql = "DELETE FROM comments WHERE id IN ($placeholders)";
            $stmt = $pdo->prepare($sql);
            $stmt->execute($comment_ids);
            
            // 提交事务
            $pdo->commit();
            
            // 使用session存储成功消息
            $_SESSION['success_message'] = '批量删除成功';
            // 构造重定向URL，保留当前的筛选条件
            $redirectUrl = 'comment_list.php';
            if ($current_status >= 0) {
                $redirectUrl .= '?status=' . $current_status;
                if (!empty($current_search)) {
                    $redirectUrl .= '&search=' . urlencode($current_search);
                }
            } elseif (!empty($current_search)) {
                $redirectUrl .= '?search=' . urlencode($current_search);
            }
            header("Location: $redirectUrl");
            exit;
        } catch (Exception $e) {
            // 回滚事务
            $pdo->rollBack();
            $error = "批量删除失败：" . $e->getMessage();
        }
    }
}

// 处理批量更改状态操作
if (isset($_POST['action']) && $_POST['action'] == 'batch_change_status') {
    if (isset($_POST['comment_ids']) && !empty($_POST['comment_ids']) && isset($_POST['status'])) {
        $comment_ids = $_POST['comment_ids'];
        $new_status = intval($_POST['status']); // 使用不同的变量名避免冲突
        $current_status = isset($_POST['status']) ? intval($_POST['status']) : -1;
        $current_search = isset($_POST['search']) ? $_POST['search'] : '';
        
        try {
            $placeholders = implode(',', array_fill(0, count($comment_ids), '?'));
            $sql = "UPDATE comments SET status = ? WHERE id IN ($placeholders)";
            $stmt = $pdo->prepare($sql);
            // 构造参数数组，第一个是状态，后面是留言ID
            $params = array_merge([$new_status], $comment_ids);
            $stmt->execute($params);
            
            // 使用session存储成功消息
            $_SESSION['success_message'] = '批量更改状态成功';
            // 构造重定向URL，保留当前的筛选条件
            $redirectUrl = 'comment_list.php';
            if ($current_status >= 0) {
                $redirectUrl .= '?status=' . $current_status;
                if (!empty($current_search)) {
                    $redirectUrl .= '&search=' . urlencode($current_search);
                }
            } elseif (!empty($current_search)) {
                $redirectUrl .= '?search=' . urlencode($current_search);
            }
            header("Location: $redirectUrl");
            exit;
        } catch (Exception $e) {
            $error = "批量更改状态失败：" . $e->getMessage();
        }
    }
}

// 获取状态参数，默认为全部
$status = isset($_GET['status']) ? intval($_GET['status']) : -1;

// 添加搜索参数处理
$search = isset($_GET['search']) ? trim($_GET['search']) : '';
$searchForSql = $search; // 保留原始搜索字符串

// 获取分页参数
$page = isset($_GET['page']) ? intval($_GET['page']) : 1;
$pageSize = 20;
$offset = ($page - 1) * $pageSize;

// 构建SQL查询
$sql = "SELECT c.*, m.title as movie_title FROM comments c LEFT JOIN movies m ON c.movie_id = m.id WHERE 1=1";
if ($status >= 0) {
    $sql .= " AND c.status = $status";
}
// 添加搜索条件
if (!empty($search)) {
    // 只在SQL查询中使用经过处理的版本
    $searchForSql = $pdo->quote('%' . $search . '%');
    $sql .= " AND (c.username LIKE $searchForSql OR c.content LIKE $searchForSql OR m.title LIKE $searchForSql)";
}

// 获取总记录数
$totalCount = $pdo->query(str_replace('c.*, m.title as movie_title', 'COUNT(*) AS count', $sql))->fetchColumn();
$totalPages = ceil($totalCount / $pageSize);

// 添加排序和分页
$sql .= " ORDER BY c.create_time DESC LIMIT $offset, $pageSize";

// 获取留言列表
$comments = $pdo->query($sql)->fetchAll();
?>
<!DOCTYPE html>
<html lang="zh-CN">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>留言管理 - 爱影视</title>
    <link rel="stylesheet" href="css/style.css">
    <script src="js/main.js"></script>
</head>
<body>
    <?php include 'includes/header.php'; ?>
    
    <div class="container">
        <div class="content-wrapper">
            <?php include 'includes/sidebar.php'; ?>
            
            <div class="content">
                <div class="main-content">
                    <div class="page-header">
                        <h2>留言管理</h2>
                        <div class="header-actions">
                            <!-- 添加搜索表单 -->
                            <form method="get" action="comment_list.php" class="search-form">
                                <input type="hidden" name="status" value="<?php echo $status; ?>">
                                <input type="text" name="search" placeholder="搜索用户名、留言内容、视频标题..." value="<?php echo htmlspecialchars($search); ?>">
                                <button type="submit" class="btn-search">搜索</button>
                            </form>
                            
                            <select onchange="location.href='comment_list.php?status='+this.value+'<?php echo !empty($search) ? '&search=' . urlencode($search) : ''; ?>'">
                                <option value="-1" <?php echo $status == -1 ? 'selected' : ''; ?>>全部状态</option>
                                <option value="1" <?php echo $status == 1 ? 'selected' : ''; ?>>已审核</option>
                                <option value="0" <?php echo $status == 0 ? 'selected' : ''; ?>>待审核</option>
                            </select>
                        </div>
                    </div>
                    
                    <?php if (!empty($error)): ?>
                        <div class="alert alert-error"><?php echo $error; ?></div>
                    <?php endif; ?>
                    
                    <?php if (!empty($success)): ?>
                        <div class="alert alert-success"><?php echo $success; ?></div>
                    <?php endif; ?>
                    
                    <form id="batch-form" method="post" action="comment_list.php">
                        <input type="hidden" name="action" id="batch-action">
                        <input type="hidden" name="status" id="batch-status-value">
                        <input type="hidden" name="status" value="<?php echo $status; ?>">
                        <input type="hidden" name="page" value="<?php echo $page; ?>">
                        <input type="hidden" name="search" value="<?php echo htmlspecialchars($search); ?>">
                        
                        <!-- 保留原有的内容区域，不修改content的子容器样式 -->
                        <table class="data-table">
                            <thead>
                                <tr>
                                    <th><input type="checkbox" id="select-all"></th>
                                    <th>ID</th>
                                    <th>用户</th>
                                    <th>留言内容</th>
                                    <th>关联视频</th>
                                    <th>IP地址</th>
                                    <th>状态</th>
                                    <th>创建时间</th>
                                    <th>操作</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if (empty($comments)): ?>
                                    <tr>
                                        <td colspan="9" class="no-data">暂无留言数据</td>
                                    </tr>
                                <?php else: ?>
                                    <?php foreach ($comments as $comment): ?>
                                        <tr>
                                            <td><input type="checkbox" name="comment_ids[]" class="comment-checkbox" value="<?php echo $comment['id']; ?>"></td>
                                            <td><?php echo $comment['id']; ?></td>
                                            <td><?php echo $comment['username']; ?></td>
                                            <td><?php echo substr($comment['content'], 0, 50); ?><?php echo strlen($comment['content']) > 50 ? '...' : ''; ?></td>
                                            <td><?php echo $comment['movie_title'] ? $comment['movie_title'] : '无关联视频'; ?></td>
                                            <td><?php echo $comment['ip_address']; ?></td>
                                            <td><?php echo $comment['status'] == 1 ? '<span class="status-active">已审核</span>' : '<span class="status-inactive">待审核</span>'; ?></td>
                                            <td><?php echo $comment['create_time']; ?></td>
                                            <td>
                                                <a href="comment_edit.php?id=<?php echo $comment['id']; ?>&status=<?php echo $status; ?>&page=<?php echo $page; ?><?php echo !empty($search) ? '&search=' . urlencode($search) : ''; ?>" class="btn-edit">编辑</a>
                                                <a href="comment_list.php?action=delete&id=<?php echo $comment['id']; ?>&status=<?php echo $status; ?>&page=<?php echo $page; ?><?php echo !empty($search) ? '&search=' . urlencode($search) : ''; ?>" class="btn-delete" onclick="return confirm('确定要删除吗？')">删除</a>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </tbody>
                        </table>
                        
                        <!-- 批量操作容器 -->
                        <?php if (!empty($comments)): ?>
                        <div class="batch-actions-container">
                            <div class="batch-actions">
                                <div class="batch-info">
                                    <label><input type="checkbox" id="select-all-bottom"> 全选</label>
                                    <span>已选择 <span id="selected-count">0</span> 项</span>
                                </div>
                                <div class="batch-controls">
                                    <select id="batch-status-select">
                                        <option value="">-- 更改状态 --</option>
                                        <option value="1">设为已审核</option>
                                        <option value="0">设为待审核</option>
                                    </select>
                                    <button id="batch-status-btn" class="btn-edit">批量更改状态</button>
                                    <button id="batch-delete-btn" class="btn-delete">批量删除</button>
                                </div>
                            </div>
                        </div>
                        <script>
                            // 获取两个全选框元素
                            const selectAllTop = document.getElementById('select-all');
                            const selectAllBottom = document.getElementById('select-all-bottom');
                            
                            // 顶部全选框事件
                            selectAllTop.addEventListener('change', function() {
                                const checked = this.checked;
                                document.querySelectorAll('.comment-checkbox').forEach(checkbox => {
                                    checkbox.checked = checked;
                                });
                                selectAllBottom.checked = checked;
                                updateSelectedCount();
                            });
                            
                            // 底部全选框事件
                            selectAllBottom.addEventListener('change', function() {
                                const checked = this.checked;
                                document.querySelectorAll('.comment-checkbox').forEach(checkbox => {
                                    checkbox.checked = checked;
                                });
                                selectAllTop.checked = checked;
                                updateSelectedCount();
                            });
                            
                            // 单个复选框更改事件
                            document.querySelectorAll('.comment-checkbox').forEach(checkbox => {
                                checkbox.addEventListener('change', function() {
                                    updateSelectedCount();
                                    // 检查是否所有复选框都被选中，同步更新全选框状态
                                    const allCheckboxes = document.querySelectorAll('.comment-checkbox');
                                    const allChecked = Array.from(allCheckboxes).every(cb => cb.checked);
                                    const anyChecked = Array.from(allCheckboxes).some(cb => cb.checked);
                                    
                                    selectAllTop.checked = allChecked;
                                    selectAllBottom.checked = allChecked;
                                    
                                    // 设置indeterminate状态（部分选中）
                                    selectAllTop.indeterminate = anyChecked && !allChecked;
                                    selectAllBottom.indeterminate = anyChecked && !allChecked;
                                });
                            });
                            
                            // 更新选中数量显示
                            function updateSelectedCount() {
                                const selectedCount = document.querySelectorAll('.comment-checkbox:checked').length;
                                document.getElementById('selected-count').textContent = selectedCount;
                            }
                            
                            // 批量更改状态
                            document.getElementById('batch-status-btn').addEventListener('click', function() {
                                const selectedStatus = document.getElementById('batch-status-select').value;
                                const selectedComments = getSelectedComments();
                                
                                if (selectedComments.length === 0) {
                                    alert('请至少选择一个留言');
                                    return;
                                }
                                
                                if (!selectedStatus) {
                                    alert('请选择要更改的状态');
                                    return;
                                }
                                
                                document.getElementById('batch-action').value = 'batch_change_status';
                                document.getElementById('batch-status-value').value = selectedStatus;
                                document.getElementById('batch-form').submit();
                            });
                            
                            // 批量删除
                            document.getElementById('batch-delete-btn').addEventListener('click', function() {
                                const selectedComments = getSelectedComments();
                                
                                if (selectedComments.length === 0) {
                                    alert('请至少选择一个留言');
                                    return;
                                }
                                
                                if (confirm('确定要删除选中的' + selectedComments.length + '条留言吗？此操作不可恢复。')) {
                                    document.getElementById('batch-action').value = 'batch_delete';
                                    document.getElementById('batch-form').submit();
                                }
                            });
                            
                            // 获取选中的留言ID
                            function getSelectedComments() {
                                const checkboxes = document.querySelectorAll('.comment-checkbox:checked');
                                const commentIds = Array.from(checkboxes).map(checkbox => checkbox.value);
                                return commentIds;
                            }
                            
                            // 初始化选中数量显示
                            window.onload = function() {
                                updateSelectedCount();
                            };
                        </script>
                        <?php endif; ?>
                    </form>
                    
                    <!-- 分页 -->
                    <?php if ($totalPages > 1): ?>
                        <div class="pagination">
                            <a href="comment_list.php?status=<?php echo $status; ?>&page=1<?php echo !empty($search) ? '&search=' . urlencode($search) : ''; ?>" <?php echo $page == 1 ? 'class="disabled"' : ''; ?>>首页</a>
                            <a href="comment_list.php?status=<?php echo $status; ?>&page=<?php echo $page - 1; ?><?php echo !empty($search) ? '&search=' . urlencode($search) : ''; ?>" <?php echo $page == 1 ? 'class="disabled"' : ''; ?>>上一页</a>
                            
                            <?php for ($i = max(1, $page - 2); $i <= min($totalPages, $page + 2); $i++): ?>
                                <a href="comment_list.php?status=<?php echo $status; ?>&page=<?php echo $i; ?><?php echo !empty($search) ? '&search=' . urlencode($search) : ''; ?>" <?php echo $page == $i ? 'class="active"' : ''; ?>><?php echo $i; ?></a>
                            <?php endfor; ?>
                            
                            <a href="comment_list.php?status=<?php echo $status; ?>&page=<?php echo $page + 1; ?><?php echo !empty($search) ? '&search=' . urlencode($search) : ''; ?>" <?php echo $page == $totalPages ? 'class="disabled"' : ''; ?>>下一页</a>
                            <a href="comment_list.php?status=<?php echo $status; ?>&page=<?php echo $totalPages; ?><?php echo !empty($search) ? '&search=' . urlencode($search) : ''; ?>" <?php echo $page == $totalPages ? 'class="disabled"' : ''; ?>>末页</a>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
    
    <?php include 'includes/footer.php'; ?>
</body>
</html>