<?php
// B站按UP主采集功能页面
require_once '../config.php';
// 添加登录检查，确保会话已初始化
require_once 'includes/check_login.php';

// 获取数据库连接
$pdo = getDbConnection();

// 初始化变量
$error = '';
$success = '';
$videos = [];

// 获取视频类型列表
$videoTypes = $pdo->query("SELECT id, name FROM video_types WHERE status = 1 ORDER BY sort_order ASC")->fetchAll();

// 获取所有分类列表并按类型分组
$categories = array();
$categoryResult = $pdo->query("SELECT id, name, type FROM movie_categories WHERE status = 1 ORDER BY type, id ASC");
while ($cat = $categoryResult->fetch(PDO::FETCH_ASSOC)) {
    if (!isset($categories[$cat['type']])) {
        $categories[$cat['type']] = array();
    }
    $categories[$cat['type']][] = $cat;
}

// 模拟正常浏览器的请求头
$browserHeaders = [
    'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36',
    'Referer: https://space.bilibili.com/',
    'Accept: application/json, text/plain, */*',
    'Accept-Language: zh-CN,zh;q=0.9',
    'Connection: keep-alive',
    'Origin: https://space.bilibili.com',
    'Cookie: ', // 将由用户输入替换
    'X-Requested-With: XMLHttpRequest',
    'Sec-Fetch-Mode: cors',
    'Sec-Fetch-Site: same-origin',
    'TE: trailers'
];

// 获取 WBI 所需的 img_key 和 sub_key
function getImgSubKeys($headers) {
    $url = 'https://api.bilibili.com/x/web-interface/nav';
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    $output = curl_exec($ch);
    curl_close($ch);

    $data = json_decode($output, true);
    if (empty($data['data']['wbi_img'])) {
        return ['error' => '获取 WBI 密钥失败，可能被风控'];
    }
    $imgUrl = $data['data']['wbi_img']['img_url'];
    $subUrl = $data['data']['wbi_img']['sub_url'];

    $imgKey = pathinfo(parse_url($imgUrl, PHP_URL_PATH), PATHINFO_FILENAME);
    $subKey = pathinfo(parse_url($subUrl, PHP_URL_PATH), PATHINFO_FILENAME);
    return ['imgKey' => $imgKey, 'subKey' => $subKey];
}

// 生成 mixin_key（签名核心）
function getMixinKey($imgKey, $subKey) {
    $mixinKeyEncTab = [
        46, 47, 18, 2, 53, 8, 23, 32, 15, 50, 10, 31, 58, 3, 45, 35, 27, 43, 5, 49,
        33, 9, 42, 19, 29, 28, 14, 39, 12, 38, 41, 13, 37, 48, 7, 16, 24, 55, 40,
        61, 26, 17, 0, 1, 60, 51, 30, 4, 22, 25, 54, 21, 56, 59, 6, 63, 57, 62, 11,
        36, 20, 34, 44, 52
    ];
    $temp = $imgKey . $subKey;
    $mixinKey = '';
    foreach ($mixinKeyEncTab as $index) {
        $mixinKey .= $temp[$index] ?? '';
    }
    return substr($mixinKey, 0, 32);
}

// 生成 WBI 签名（w_rid 和 wts）
function getWbiSign($params, $imgKey, $subKey) {
    $mixinKey = getMixinKey($imgKey, $subKey);
    $params['wts'] = time(); // 当前时间戳（秒级，必须）
    
    ksort($params); // 参数按 key 升序排序（签名必要步骤）
    $query = http_build_query($params); // 拼接成 URL 参数字符串
    $wRid = md5($query . $mixinKey); // 计算 MD5 签名
    return [$params, $wRid];
}

// 分页获取所有视频
function getAllVideos($mid, $headers, $pn = 1) {
    $ps = 42; // B站强制限制：每页最大42条
    $retryCount = 3; // 失败重试次数
    $retryDelay = 1; // 重试间隔（秒）
    
    while ($retryCount > 0) {
        try {
            // 1. 先获取 WBI 密钥
            $wbiKeys = getImgSubKeys($headers);
            if (isset($wbiKeys['error'])) {
                throw new Exception($wbiKeys['error']);
            }
            $imgKey = $wbiKeys['imgKey'];
            $subKey = $wbiKeys['subKey'];

            // 2. 构造请求参数
            $params = [
                'mid' => $mid,
                'ps' => $ps,
                'pn' => $pn,
                'tid' => 0, // 0 = 全部分区
                'keyword' => '',
                'order' => 'pubdate', // 按发布时间排序
                'platform' => 'web',
                'web_location' => 1550101,
                'order_avoided' => 'true'
            ];

            // 3. 生成 WBI 签名
            list($params, $wRid) = getWbiSign($params, $imgKey, $subKey);

            // 4. 发送请求（关键：添加请求头）
            $url = "https://api.bilibili.com/x/space/wbi/arc/search?w_rid={$wRid}";
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $url . '&' . http_build_query($params));
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
            curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
            curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
            curl_setopt($ch, CURLOPT_FOLLOWLOCATION, false);
            curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10);
            curl_setopt($ch, CURLOPT_TIMEOUT, 15);
            
            // 增加IP轮转相关设置（如果有代理的话）
            // curl_setopt($ch, CURLOPT_PROXY, $proxy); // 可选：代理服务器
            
            $output = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE); // 获取响应状态码
            
            // 检查是否有curl错误
            if (curl_errno($ch)) {
                throw new Exception('CURL错误: ' . curl_error($ch));
            }
            
            curl_close($ch);

            // 5. 校验响应状态
            if ($httpCode != 200) {
                // 特殊处理412状态码，可能是请求头不完整导致的
                if ($httpCode == 412) {
                    throw new Exception("请求被拦截（HTTP 412），请检查Cookie是否有效或尝试使用不同的Cookie");
                }
                throw new Exception("请求被拦截，HTTP状态码：{$httpCode}");
            }
            
            // 检查响应是否为空
            if (empty($output)) {
                throw new Exception('API返回空响应');
            }
            
            $data = json_decode($output, true);
            
            // 检查JSON解析是否成功
            if (json_last_error() !== JSON_ERROR_NONE) {
                throw new Exception('响应解析失败: ' . json_last_error_msg());
            }
            
            if ($data['code'] != 0) {
                throw new Exception("获取失败: {$data['message']}（错误码：{$data['code']}）");
            }

            // 6. 提取视频数据
            $videos = $data['data']['list']['vlist'];
            if (empty($videos)) {
                return []; // 没有视频，返回空数组
            }
            
            return $videos;
            
        } catch (Exception $e) {
            $retryCount--;
            if ($retryCount <= 0) {
                return ['error' => $e->getMessage()];
            }
            // 延迟后重试
            sleep($retryDelay);
            // 递增重试延迟时间（指数退避）
            $retryDelay *= 2;
        }
    }
    
    return ['error' => '达到最大重试次数，获取视频失败'];
}

// 保存视频数据到数据库
function saveVideosToDatabase($videos, $selectedType, $selectedCategory, $pdo) {
    $totalSaved = 0;
    $totalSkipped = 0;
    
    foreach ($videos as $video) {
        try {
            $title = $video['title'] ?? '未命名视频';
            $description = '<p>'.$video['description'] ?? ''.'</p>';  
            $pic = $video['pic'] ?? '';
            $author = $video['author'] ?? '';
            $bvid = $video['bvid'] ?? '';
            $subtitle = $video['subtitle'];
            $actors = '';
            $region = '';
            $language = '';
            $release_year = date('Y', $video['created']) ?? '';
            $play_url = '';
            
            // 确保所有值都是字符串类型
            $title = (string)$title;
            $description = (string)$description;
            $pic = (string)$pic;
            $author = (string)$author;
            $bvid = (string)$bvid;
            $subtitle = (string)$subtitle;
            $actors = (string)$actors;
            $region = (string)$region;
            $language = (string)$language;
            $play_url = (string)$play_url;
            
            // 检查电影是否已存在
            $stmt = $pdo->prepare("SELECT id FROM movies WHERE title = :title LIMIT 1");
            $stmt->execute(['title' => $title]);
            $existingMovie = $stmt->fetch();
            
            if ($existingMovie) {
                $totalSkipped++;
                continue;
            }
            
            // 插入movies表，添加所有字段
            $stmt = $pdo->prepare("INSERT INTO movies (title, subtitle, description, cover_img, director, type, category_id, status, total_episodes, create_time, actors, region, language, release_year, play_url) VALUES (:title, :subtitle, :description, :cover_img, :director, :type, :category_id, 1, 1, NOW(), :actors, :region, :language, :release_year, :play_url)");
            $stmt->execute([
                'title' => $title,
                'subtitle' => $subtitle,
                'description' => $description,
                'cover_img' => $pic,
                'director' => $author,
                'type' => $selectedType,
                'category_id' => $selectedCategory,
                'actors' => $actors,
                'region' => $region,
                'language' => $language,
                'release_year' => $release_year,
                'play_url' => $play_url
            ]);
            
            $movieId = $pdo->lastInsertId();
            
            // 插入episodes表
            $playUrl = 'bilibili/index.php?url=' . $bvid;
            $stmt = $pdo->prepare("INSERT INTO episodes (movie_id, episode_num, episode_title, play_url, create_time) VALUES (:movie_id, 1, :episode_title, :play_url, NOW())");
            $stmt->execute([
                'movie_id' => $movieId,
                'episode_title' => $title,
                'play_url' => $playUrl
            ]);
            
            $totalSaved++;
        } catch (Exception $e) {
            // 单个视频保存失败不影响其他视频
            continue;
        }
    }
    
    return [
        'saved' => $totalSaved,
        'skipped' => $totalSkipped
    ];
}

// 处理表单提交
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['collect_up'])) {
    try {
        // 获取表单数据
        $userCookie = trim($_POST['user_cookie']);
        $upUrl = trim($_POST['up_url']);
        $pageNum = intval($_POST['page_num']);
        $selectedType = intval($_POST['video_type']);
        $selectedCategory = intval($_POST['video_category']);
        
        // 验证必填字段
        if (empty($userCookie)) {
            throw new Exception('请输入用户Cookie');
        }
        
        if (empty($upUrl)) {
            throw new Exception('请输入UP主空间页地址');
        }
        
        if ($selectedType <= 0) {
            throw new Exception('请选择视频类型');
        }
        
        if ($selectedCategory <= 0) {
            throw new Exception('请选择视频分类');
        }
        
        // 从UP主空间地址提取MID
        preg_match('/space\.bilibili\.com\/(\d+)/', $upUrl, $matches);
        if (empty($matches[1])) {
            throw new Exception('无法从地址中提取UP主MID，请检查地址格式是否正确');
        }
        
        $upMid = $matches[1];
        
        // 更新请求头中的Cookie
        $headers = $browserHeaders;
        foreach ($headers as $key => $header) {
            if (strpos($header, 'Cookie:') === 0) {
                $headers[$key] = 'Cookie: ' . $userCookie;
                break;
            }
        }
        
        // 获取视频数据
        $videos = getAllVideos($upMid, $headers, $pageNum);
        
        if (isset($videos['error'])) {
            throw new Exception($videos['error']);
        }
        
        if (empty($videos)) {
            throw new Exception('未获取到视频数据');
        }
        
        // 保存到数据库
        $result = saveVideosToDatabase($videos, $selectedType, $selectedCategory, $pdo);
        
        $success = "成功保存 {$result['saved']} 个视频，跳过 {$result['skipped']} 个已存在的视频";
        
    } catch (Exception $e) {
        $error = $e->getMessage();
    }
}
?>

<!DOCTYPE html>
<html lang="zh-CN">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>B站按UP主采集 - 后台管理</title>
    <link rel="stylesheet" href="css/style.css">
    <script src="https://registry.npmmirror.com/axios/1.12.2/files/dist/axios.min.js"></script>
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            // 类型选择变化时，更新分类列表
            const videoTypeSelect = document.getElementById('video_type');
            const videoCategorySelect = document.getElementById('video_category');
            
            function updateCategories() {
                const selectedType = videoTypeSelect.value;
                const categoryGroups = <?php echo json_encode($categories); ?>;
                
                // 清空现有选项
                videoCategorySelect.innerHTML = '<option value="0">请选择分类</option>';
                
                // 添加对应类型的分类
                if (selectedType && categoryGroups[selectedType]) {
                    categoryGroups[selectedType].forEach(category => {
                        const option = document.createElement('option');
                        option.value = category.id;
                        option.textContent = category.name;
                        videoCategorySelect.appendChild(option);
                    });
                }
            }
            
            // 初始更新分类
            updateCategories();
            
            // 绑定类型变化事件
            videoTypeSelect.addEventListener('change', updateCategories);
        });
    </script>
</head>
<body>
    <?php include 'includes/header.php'; ?>
    
    <div class="container">
        <div class="content-wrapper">
            <?php include 'includes/sidebar.php'; ?>
            
            <div class="content">
                <div class="main-content">
                    <div class="breadcrumb">
                        <a href="index.php">首页</a> &gt; 
                        <span>B站按UP主采集</span>
                    </div>
                    
                    <div class="panel">
                        <div class="panel-header">
                            <h3>B站按UP主采集</h3>
                        </div>
                        <div class="panel-body">
                            
                            <?php if ($error): ?>
                                <div class="alert alert-error">
                                    <?php echo $error; ?>
                                </div>
                            <?php endif; ?>
                            
                            <?php if ($success): ?>
                                <div class="alert alert-success">
                                    <?php echo $success; ?>
                                </div>
                            <?php endif; ?>
                            
                            <form method="post" action="get_up_videos.php">
                                <div class="form-group">
                                    <label for="user_cookie">用户Cookie</label>
                                    <textarea id="user_cookie" name="user_cookie" rows="3" class="form-control" placeholder="请输入B站用户Cookie"></textarea>
                                    <p class="help-text">Cookie用于访问UP主的视频数据，建议使用已登录账号的Cookie</p>
                                    <div class="form-actions" style="margin-top: 5px;">
                                        <button type="button" id="clear_cookie" class="btn btn-default">清除Cookie</button>
                                    </div>
                                </div>
                                
                                <div class="form-group">
                                    <label for="up_url">UP主空间页地址</label>
                                    <input type="text" id="up_url" name="up_url" class="form-control" placeholder="例如：https://space.bilibili.com/12345678">
                                </div>
                                
                                <div class="form-group">
                                    <label for="page_num">视频页码(pn)</label>
                                    <input type="number" id="page_num" name="page_num" class="form-control" value="1" min="1" step="1">
                                    <p class="help-text">请输入要获取的视频页码，每页获取42条视频，默认第1页</p>
                                </div>
                                
                                <div class="form-group">
                                    <label for="video_type">视频类型</label>
                                    <select id="video_type" name="video_type" class="form-control">
                                        <option value="0">请选择类型</option>
                                        <?php foreach ($videoTypes as $type): ?>
                                            <option value="<?php echo $type['id']; ?>"><?php echo $type['name']; ?></option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                                
                                <div class="form-group">
                                    <label for="video_category">视频分类</label>
                                    <select id="video_category" name="video_category" class="form-control">
                                        <option value="0">请选择分类</option>
                                    </select>
                                </div>
                                
                                <div class="form-actions">
                                    <button type="submit" name="collect_up" class="btn btn-primary">开始采集</button>
                                </div>
                            </form>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <?php include 'includes/footer.php'; ?>
    
    <script>
        // 加载保存的Cookie值和表单验证
        document.addEventListener('DOMContentLoaded', function() {
            const cookieInput = document.getElementById('user_cookie');
            const clearCookieBtn = document.getElementById('clear_cookie');
            const pageNumInput = document.getElementById('page_num');
            const collectForm = document.querySelector('form');
            
            // 尝试从localStorage加载保存的Cookie
            const savedCookie = localStorage.getItem('bilibili_cookie');
            if (savedCookie) {
                cookieInput.value = savedCookie;
            }
            
            // 监听输入变化，自动保存到localStorage
            cookieInput.addEventListener('input', function() {
                localStorage.setItem('bilibili_cookie', this.value);
            });
            
            // 清除Cookie按钮点击事件
            clearCookieBtn.addEventListener('click', function() {
                cookieInput.value = '';
                localStorage.removeItem('bilibili_cookie');
                alert('Cookie已清除');
            });
            
            // 表单提交验证
            collectForm.addEventListener('submit', function(e) {
                // 验证页码输入
                const pageNum = parseInt(pageNumInput.value);
                if (isNaN(pageNum) || pageNum < 1) {
                    alert('请输入有效的页码（正整数）');
                    e.preventDefault();
                    pageNumInput.focus();
                    return false;
                }
            });
        });
    </script>
</body>
</html>