<?php
// 引入登录检查文件
require_once('includes/check_login.php');

// 引入数据库配置
require_once('../config.php');

// 初始化变量
$movie_id = '';
$title = '';
$subtitle = '';
$type = '';
$category_id = '';
$director = '';
$actors = '';
$region = '';
$language = '';
$release_year = '';
$cover_img = '';
$description = '';
$play_url = '';
$total_episodes = 1;  // 默认设置为1
$status = 1;  // 在添加模式下默认为上架状态
$error = '';
$success = '';
$episodes = array();

// 数据库连接
$pdo = getDbConnection();

// 获取视频类型列表
$videoTypesByCode = array(); 
$typeIdToCodeMap = array(); 
$sql = "SELECT id, name, code FROM video_types WHERE status = 1 ORDER BY sort_order ASC";
$result = $pdo->query($sql);
if ($result && $result->rowCount() > 0) {
    while ($row = $result->fetch(PDO::FETCH_ASSOC)) {
        $videoTypesByCode[$row['code']] = $row;
        $typeIdToCodeMap[$row['id']] = $row['code'];
    }
}

// 获取分类列表
$allCategories = array(); 
$sql = "SELECT id, name, type FROM movie_categories WHERE status = 1 ORDER BY id ASC";
$result = $pdo->query($sql);
if ($result && $result->rowCount() > 0) {
    while ($row = $result->fetch(PDO::FETCH_ASSOC)) {
        // 添加type_str字段以便JavaScript正确匹配
        $row['type_str'] = isset($typeIdToCodeMap[$row['type']]) ? $typeIdToCodeMap[$row['type']] : 'other';
        $allCategories[] = $row; 
    }
}

// 判断是编辑模式还是添加模式
$isEditMode = isset($_GET['id']);

// 编辑模式：从数据库获取视频信息
if ($isEditMode) {
    $movie_id = intval($_GET['id']);
    
    $sql = "SELECT * FROM movies WHERE id = ?";
    $stmt = $pdo->prepare($sql);
    $stmt->bindParam(1, $movie_id);
    $stmt->execute();
    
    if ($stmt->rowCount() > 0) {
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        $title = $row['title'];
        $subtitle = $row['subtitle'];
        $type = isset($typeIdToCodeMap[$row['type']]) ? $typeIdToCodeMap[$row['type']] : 'other';
        $category_id = $row['category_id'];
        $director = $row['director'];
        $actors = $row['actors'];
        $region = $row['region'];
        $language = $row['language'];
        $release_year = $row['release_year'];
        $cover_img = $row['cover_img'];
        $description = $row['description'];
        $play_url = $row['play_url'];
        // 先使用数据库中的总集数
        $total_episodes = $row['total_episodes'];
        $status = $row['status'];
        
        // 获取分集信息
        $sql = "SELECT * FROM episodes WHERE movie_id = ? ORDER BY episode_num ASC";
        $stmt = $pdo->prepare($sql);
        $stmt->bindParam(1, $movie_id);
        $stmt->execute();
        $episodes = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // 重要：根据实际剧集数量更新总集数
        if (!empty($episodes)) {
            // 使用实际的剧集数量来更新总集数
            $actualEpisodeCount = count($episodes);
            // 如果实际剧集数与数据库中的总集数不一致，更新数据库
            if ($actualEpisodeCount != $total_episodes) {
                $total_episodes = $actualEpisodeCount;
                // 更新数据库中的总集数
                $updateSql = "UPDATE movies SET total_episodes = ? WHERE id = ?";
                $updateStmt = $pdo->prepare($updateSql);
                $updateStmt->execute([$total_episodes, $movie_id]);
            }
        }
    } else {
        $error = "视频不存在";
    }
}

// 处理分集信息的函数
function saveEpisodes($pdo, $movieId, $totalEpisodes, $episodeUrls, $episodeTitles = array()) {
    // 先删除旧的分集信息
    $sql = "DELETE FROM episodes WHERE movie_id = ?";
    $stmt = $pdo->prepare($sql);
    $stmt->execute([$movieId]);
    
    // 实际插入的剧集数
    $actualInsertedEpisodes = 0;
    
    // 处理分集信息
    if ($totalEpisodes > 1 && is_array($episodeUrls)) {
        for ($i = 0; $i < $totalEpisodes; $i++) {
            if (isset($episodeUrls[$i]) && !empty($episodeUrls[$i])) {
                $episodeNum = $i + 1;
                $episodeTitle = isset($episodeTitles[$i]) ? trim($episodeTitles[$i]) : '';
                
                // 根据是否有标题选择不同的SQL语句
                if (!empty($episodeTitle)) {
                    $sql = "INSERT INTO episodes (movie_id, episode_num, episode_title, play_url, create_time) VALUES (?, ?, ?, ?, NOW())";
                    $stmt = $pdo->prepare($sql);
                    $stmt->execute([$movieId, $episodeNum, $episodeTitle, $episodeUrls[$i]]);
                } else {
                    $sql = "INSERT INTO episodes (movie_id, episode_num, play_url, create_time) VALUES (?, ?, ?, NOW())";
                    $stmt = $pdo->prepare($sql);
                    $stmt->execute([$movieId, $episodeNum, $episodeUrls[$i]]);
                }
                
                $actualInsertedEpisodes++;
            }
        }
    }
    
    // 重要：更新movies表中的总集数为实际插入的剧集数
    if ($actualInsertedEpisodes > 0) {
        $updateSql = "UPDATE movies SET total_episodes = ? WHERE id = ?";
        $updateStmt = $pdo->prepare($updateSql);
        $updateStmt->execute([$actualInsertedEpisodes, $movieId]);
    } else if ($totalEpisodes > 1) {
        // 如果应该有多集但没有插入任何剧集，默认设置为1
        $updateSql = "UPDATE movies SET total_episodes = 1 WHERE id = ?";
        $updateStmt = $pdo->prepare($updateSql);
        $updateStmt->execute([$movieId]);
    }
}

// 处理表单提交
if ($_SERVER['REQUEST_METHOD'] == 'POST' && empty($error)) {
    // 获取表单数据
    $title = $_POST['title'];
    $subtitle = $_POST['subtitle'];
    $type = $_POST['type'];
    $category_id = $_POST['category_id'];
    $director = $_POST['director'];
    $actors = $_POST['actors'];
    $region = $_POST['region'];
    $language = $_POST['language'];
    // 修改release_year的处理逻辑，添加验证和合理的默认值
    $release_year = !empty($_POST['release_year']) ? intval($_POST['release_year']) : date('Y');
    // 确保年份在有效范围内（1901-2155）
    if ($release_year < 1901 || $release_year > 2155) {
        $release_year = date('Y'); // 如果超出范围，使用当前年份
    }
    $cover_img = $_POST['cover_img'];
    $description = $_POST['description'];
    $play_url = $_POST['play_url'];
    // 确保总集数至少为1
    $total_episodes = isset($_POST['total_episodes']) ? max(1, intval($_POST['total_episodes'])) : 1;
    $status = $_POST['status'];
    $episodeUrls = isset($_POST['episode_urls']) ? $_POST['episode_urls'] : [];
    $episodeTitles = isset($_POST['episode_titles']) ? $_POST['episode_titles'] : [];
    
    // 验证数据
    if (empty($title)) {
        $error = "请输入视频标题";
    } elseif (empty($type) || !isset($videoTypesByCode[$type])) {
        $error = "请选择有效的视频类型";
    } elseif (empty($category_id)) {
        $error = "请选择分类";
    } elseif (empty($cover_img)) {
        $error = "请输入封面图片URL";
    } elseif ($total_episodes == 1 && empty($play_url)) {
        $error = "请输入视频URL";
    } elseif ($total_episodes > 1 && (empty($episodeUrls) || !is_array($episodeUrls))) {
        $error = "请填写分集播放地址";
    } else {
        // 确保类型值与数据库保持一致
        $typeValue = null;
        if (isset($videoTypesByCode[$type])) {
            $typeValue = $videoTypesByCode[$type]['id'];
        } else {
            $error = "无效的视频类型";
        }
        
        if (empty($error)) {
            if ($isEditMode) {
                // 编辑模式：更新视频数据
                try {
                    // 开始事务
                    $pdo->beginTransaction();
                    
                    $sql = "UPDATE movies SET title = ?, subtitle = ?, type = ?, category_id = ?, director = ?, actors = ?, region = ?, language = ?, release_year = ?, cover_img = ?, description = ?, play_url = ?, total_episodes = ?, status = ? WHERE id = ?";
                    $stmt = $pdo->prepare($sql);
                    
                    $stmt->execute([$title, $subtitle, $typeValue, $category_id, $director, $actors, $region, $language, $release_year, $cover_img, $description, $play_url, $total_episodes, $status, $movie_id]);
                    
                    // 调用函数保存分集信息
                    saveEpisodes($pdo, $movie_id, $total_episodes, $episodeUrls, $episodeTitles);
                    
                    // 提交事务
                    $pdo->commit();
                    
                    // 重新从数据库获取最新数据
                    $sql = "SELECT * FROM movies WHERE id = ?";
                    $stmt = $pdo->prepare($sql);
                    $stmt->execute([$movie_id]);
                    
                    if ($stmt->rowCount() > 0) {
                        $row = $stmt->fetch(PDO::FETCH_ASSOC);
                        $title = $row['title'];
                        $subtitle = $row['subtitle'];
                        $type = isset($typeIdToCodeMap[$row['type']]) ? $typeIdToCodeMap[$row['type']] : 'other';
                        $category_id = $row['category_id'];
                        $director = $row['director'];
                        $actors = $row['actors'];
                        $region = $row['region'];
                        $language = $row['language'];
                        $release_year = $row['release_year'];
                        $cover_img = $row['cover_img'];
                        $description = $row['description'];
                        $play_url = $row['play_url'];
                        $total_episodes = $row['total_episodes'];
                        $status = $row['status'];
                        
                        // 同时更新分集信息
                        $sql = "SELECT * FROM episodes WHERE movie_id = ? ORDER BY episode_num ASC";
                        $stmt = $pdo->prepare($sql);
                        $stmt->execute([$movie_id]);
                        $episodes = $stmt->fetchAll(PDO::FETCH_ASSOC);
                    }
                    
                    $success = "视频更新成功";
                } catch (PDOException $e) {
                    // 回滚事务
                    $pdo->rollBack();
                    $error = "更新失败：" . $e->getMessage();
                }
            } else {
                // 添加模式：插入新视频数据
                try {
                    // 开始事务
                    $pdo->beginTransaction();
                    
                    $sql = "INSERT INTO movies (title, subtitle, type, category_id, director, actors, region, language, release_year, cover_img, description, play_url, total_episodes, status, create_time) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())";
                    $stmt = $pdo->prepare($sql);
                    
                    $stmt->execute([$title, $subtitle, $typeValue, $category_id, $director, $actors, $region, $language, $release_year, $cover_img, $description, $play_url, $total_episodes, $status]);
                    
                    $newMovieId = $pdo->lastInsertId();
                    
                    // 调用函数保存分集信息
                    saveEpisodes($pdo, $newMovieId, $total_episodes, $episodeUrls, $episodeTitles);
                    
                    // 提交事务
                    $pdo->commit();
                    
                    $success = "视频添加成功";
                    // 重置表单（添加模式下）
                    $title = '';
                    $subtitle = '';
                    $type = '';
                    $category_id = '';
                    $director = '';
                    $actors = '';
                    $region = '';
                    $language = '';
                    $release_year = '';
                    $cover_img = '';
                    $description = '';
                    $play_url = '';
                    $total_episodes = 1;
                    $status = 1;
                    $episodes = array();
                } catch (PDOException $e) {
                    // 回滚事务
                    $pdo->rollBack();
                    $error = "添加失败：" . $e->getMessage();
                }
            }
        }
    }
}
?>
<!DOCTYPE html>
<html lang="zh-CN">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $isEditMode ? '编辑视频' : '添加视频'; ?> - 后台管理</title>
    <link rel="stylesheet" href="css/style.css">
    <script src="js/main.js"></script>
    <script>
        // 当视频类型改变时，更新分类下拉框
        function updateCategories() {
            const type = document.getElementById('type').value;
            const categorySelect = document.getElementById('category_id');
            const videoUrlContainer = document.getElementById('video_url_container');
            const episodesFields = document.getElementById('episodes_fields');
            const totalEpisodesInput = document.getElementById('total_episodes');
            const currentCategoryId = <?php echo json_encode($category_id); ?>;
            
            // 清空当前分类选项
            categorySelect.innerHTML = '<option value="">请选择分类</option>';
            
            // 获取页面中的分类数据
            const categories = JSON.parse(document.getElementById('categories-data').value);
            
            // 添加对应类型的分类（使用type_str字段）
            categories.forEach(category => {
                if (category.type_str === type) {
                    const option = document.createElement('option');
                    option.value = category.id;
                    option.textContent = category.name;
                    if (category.id == currentCategoryId) {
                        option.selected = true;
                    }
                    categorySelect.appendChild(option);
                }
            });
            
            // 根据总集数来决定显示视频URL还是分集字段
            const totalEpisodes = parseInt(totalEpisodesInput.value) || 1;
            updateEpisodeDisplay(totalEpisodes, videoUrlContainer, episodesFields);
        }
        
        // 处理总集数变化时的显示逻辑
        function updateEpisodeDisplay(totalEpisodes, videoUrlContainer, episodesFields) {
            videoUrlContainer.style.display = totalEpisodes > 1 ? 'none' : 'block';
            
            if (totalEpisodes > 1) {
                generateEpisodeFields();
            } else {
                episodesFields.innerHTML = '';
            }
        }
        
        // 生成分集输入字段
        function generateEpisodeFields() {
            const totalEpisodes = parseInt(document.getElementById('total_episodes').value) || 0;
            const episodesFieldsContainer = document.getElementById('episodes_fields');
            const existingEpisodes = <?php echo json_encode($episodes); ?>
            
            // 保存用户已经输入的最新值
            const currentEpisodeValues = {};
            const currentEpisodeTitles = {};
            
            // 获取现有URL值
            const existingEpisodeInputs = episodesFieldsContainer.querySelectorAll('input[name="episode_urls[]"]');
            existingEpisodeInputs.forEach((input, index) => {
                if (input.value) {
                    currentEpisodeValues[index + 1] = input.value; // 使用集数作为键
                }
            });
            
            // 获取现有标题值
            const existingTitleInputs = episodesFieldsContainer.querySelectorAll('input[name="episode_titles[]"]');
            existingTitleInputs.forEach((input, index) => {
                if (input.value) {
                    currentEpisodeTitles[index + 1] = input.value; // 使用集数作为键
                }
            });
            
            // 清空现有字段
            episodesFieldsContainer.innerHTML = '';
            
            // 生成新的分集字段
            for (let i = 0; i < totalEpisodes; i++) {
                const episodeNum = i + 1;
                const episodeDiv = document.createElement('div');
                // 恢复原始的多集输入框样式类
                episodeDiv.className = 'form-group';
                
                // 优先使用用户当前输入的值，如果没有则使用现有分集信息
                let episodeUrl = currentEpisodeValues[episodeNum] || '';
                let episodeTitle = currentEpisodeTitles[episodeNum] || '';
                
                if (!episodeUrl || !episodeTitle) {
                    for (let j = 0; j < existingEpisodes.length; j++) {
                        if (existingEpisodes[j].episode_num === episodeNum) {
                            if (!episodeUrl) episodeUrl = existingEpisodes[j].play_url || '';
                            if (!episodeTitle && existingEpisodes[j].episode_title) {
                                episodeTitle = existingEpisodes[j].episode_title || '';
                            }
                            break;
                        }
                    }
                }
                
                episodeDiv.innerHTML = `
                    <h4 class="mb-2">第${episodeNum}集</h4>
                    <label for="episode_title_${i}">分集标题</label>
                    <input type="text" id="episode_title_${i}" name="episode_titles[]" value="${episodeTitle}" placeholder="请输入第${episodeNum}集的标题（可选）">
                    <label for="episode_url_${i}">播放地址 <span class="required">*</span></label>
                    <input type="text" id="episode_url_${i}" name="episode_urls[]" value="${episodeUrl}" placeholder="请输入第${episodeNum}集的播放地址">
                `;
                
                episodesFieldsContainer.appendChild(episodeDiv);
            }
        }
        
        // 页面加载完成后执行
        document.addEventListener('DOMContentLoaded', function() {
            // 如果总集数为空或0，设置默认值为1
            const totalEpisodesInput = document.getElementById('total_episodes');
            const videoUrlContainer = document.getElementById('video_url_container');
            const episodesFields = document.getElementById('episodes_fields');
            
            if (!totalEpisodesInput.value || parseInt(totalEpisodesInput.value) <= 0) {
                totalEpisodesInput.value = 1;
            }
            
            updateCategories();
            
            // 为总集数输入框添加事件监听 - 修改为只处理集数显示逻辑
            totalEpisodesInput.addEventListener('input', function() {
                // 确保总集数不能小于1
                const value = parseInt(totalEpisodesInput.value) || 0;
                if (value < 1) {
                    totalEpisodesInput.value = 1;
                }
                
                const totalEpisodes = parseInt(totalEpisodesInput.value) || 1;
                updateEpisodeDisplay(totalEpisodes, videoUrlContainer, episodesFields);
            });
        });
    </script>
</head>
<body>
    <?php include('includes/header.php'); ?>
    
    <div class="container">
        <div class="content-wrapper">
            <?php include('includes/sidebar.php'); ?>
            
            <div class="content">
                <div class="main-content">
                    <div class="breadcrumb">
                        <a href="index.php">首页</a> &gt; 
                        <a href="movie_list.php">视频管理</a> &gt; 
                        <span><?php echo $isEditMode ? '编辑视频' : '添加视频'; ?></span>
                    </div>
                    
                    <!-- 保留原有的内容区域，不修改content中的子容器样式 -->
                    <div class="panel">
                        <div class="panel-header">
                            <h2><?php echo $isEditMode ? '编辑视频' : '添加视频'; ?></h2>
                        </div>
                        
                        <div class="panel-body">
                            <!-- 显示错误消息 -->
                            <?php if (!empty($error)): ?>
                                <div class="alert alert-error">
                                    <?php echo $error; ?>
                                </div>
                            <?php endif; ?>
                            
                            <!-- 显示成功消息 -->
                            <?php if (!empty($success)): ?>
                                <div class="alert alert-success">
                                    <?php echo $success; ?>
                                </div>
                            <?php endif; ?>
                            
                            <!-- 如果视频不存在（编辑模式），则不显示表单 -->
                            <?php if ($isEditMode && !empty($error)): ?>
                                <div class="alert alert-info">
                                    <a href="movie_list.php" class="btn btn-primary">返回视频列表</a>
                                </div>
                            <?php else: ?>
                                <form action="<?php echo $isEditMode ? "movie_edit.php?id=$movie_id" : "movie_edit.php"; ?>" method="post">
                                    <input type="hidden" id="categories-data" value='<?php echo json_encode($allCategories); ?>'>
                                    
                                    <div class="form-group">
                                        <label for="title">视频标题 <span class="required">*</span></label>
                                        <input type="text" id="title" name="title" value="<?php echo htmlspecialchars($title); ?>" placeholder="请输入视频标题">
                                    </div>
                                 
                                    <div class="form-group">
                                        <label for="subtitle">副标题</label>
                                        <input type="text" id="subtitle" name="subtitle" value="<?php echo htmlspecialchars($subtitle); ?>" placeholder="请输入副标题">
                                    </div>

                                    <div class="form-group">
                                        <label for="type">视频类型 <span class="required">*</span></label>
                                        <select id="type" name="type" onchange="updateCategories()">
                                            <option value="">请选择类型</option>
                                            <?php foreach ($videoTypesByCode as $typeData): ?>
                                                <option value="<?php echo $typeData['code']; ?>" <?php if ($type == $typeData['code']) echo 'selected'; ?>><?php echo $typeData['name']; ?></option>
                                            <?php endforeach; ?>
                                        </select>
                                    </div>
                                 
                                    <div class="form-group">
                                        <label for="category_id">分类 <span class="required">*</span></label>
                                        <select id="category_id" name="category_id">
                                            <option value="">请选择分类</option>
                                        </select>
                                    </div>
                                 
                                    <div class="form-group">
                                        <label for="director">导演</label>
                                        <input type="text" id="director" name="director" value="<?php echo htmlspecialchars($director); ?>" placeholder="请输入导演信息">
                                    </div>
                                 
                                    <div class="form-group">
                                        <label for="actors">演员</label>
                                        <input type="text" id="actors" name="actors" value="<?php echo htmlspecialchars($actors); ?>" placeholder="请输入演员信息，多个演员用逗号分隔">
                                    </div>
                                 
                                    <div class="form-group">
                                        <label for="region">地区</label>
                                        <input type="text" id="region" name="region" value="<?php echo htmlspecialchars($region); ?>" placeholder="请输入地区信息">
                                    </div>
                                 
                                    <div class="form-group">
                                        <label for="language">语言</label>
                                        <input type="text" id="language" name="language" value="<?php echo htmlspecialchars($language); ?>" placeholder="请输入语言信息">
                                    </div>
                                 
                                    <div class="form-group">
                                        <label for="release_year">上映年份</label>
                                        <input type="number" id="release_year" name="release_year" value="<?php echo $release_year; ?>" placeholder="请输入上映年份">
                                    </div>
                                 
                                    <div class="form-group">
                                        <label for="cover_img">封面图片 <span class="required">*</span></label>
                                        <input type="text" id="cover_img" name="cover_img" value="<?php echo htmlspecialchars($cover_img); ?>" placeholder="请输入封面图片URL">
                                    </div>
                                 
                                    <div class="form-group">
                                        <label for="description">剧情简介</label>
                                        <textarea id="description" name="description" rows="5" placeholder="请输入剧情简介"><?php echo htmlspecialchars($description); ?></textarea>
                                    </div>
                                 
                                    <!-- 剧集集数 -->
                                    <div id="episodes_container" class="form-group">
                                        <label for="total_episodes">总集数 <span class="required">*</span></label>
                                        <input type="number" id="total_episodes" name="total_episodes" min="1" value="<?php echo $total_episodes; ?>" placeholder="请输入总集数">
                                    </div>
                                 
                                    <!-- 电影URL -->
                                    <div id="video_url_container" class="form-group">
                                        <label for="play_url">视频URL <span class="required">*</span></label>
                                        <input type="text" id="play_url" name="play_url" value="<?php echo htmlspecialchars($play_url); ?>" placeholder="请输入视频URL">
                                    </div>
                                 
                                    <!-- 分集字段 -->
                                    <div id="episodes_fields"></div>
                                 
                                    <div class="form-group">
                                        <label for="status">状态</label>
                                        <select id="status" name="status">
                                            <option value="1" <?php if ($status == 1) echo 'selected'; ?>>上架</option>
                                            <option value="0" <?php if ($status == 0) echo 'selected'; ?>>下架</option>
                                        </select>
                                    </div>
                                 
                                    <div class="form-actions">
                                        <button type="submit" class="btn btn-primary"><?php echo $isEditMode ? '保存更改' : '添加视频'; ?></button>
                                        <a href="movie_list.php" class="btn btn-default">返回列表</a>
                                    </div>
                                </form>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <?php include('includes/footer.php'); ?>
</body>
</html>
