<?php
// 视频列表页面
require_once 'includes/check_login.php';
require_once '../config.php';

// 初始化session，用于存储操作结果消息
session_start();

// 在文件顶部，数据库连接代码后添加处理删除操作的代码
// 获取数据库连接
$pdo = getDbConnection();

// 处理删除操作
$error = '';
$success = '';
// 检查session中的成功消息
if (isset($_SESSION['success_message']) && !empty($_SESSION['success_message'])) {
    $success = $_SESSION['success_message'];
    // 显示后清除session中的消息
    unset($_SESSION['success_message']);
}
if (isset($_GET['action']) && $_GET['action'] == 'delete' && isset($_GET['id'])) {
    $movie_id = intval($_GET['id']);
    
    try {
        // 开启事务
        $pdo->beginTransaction();
        
        // 先删除相关留言
        $sql = "DELETE FROM comments WHERE movie_id = ?";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([$movie_id]);
        
        // 然后删除视频
        $sql = "DELETE FROM movies WHERE id = ?";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([$movie_id]);
        
        // 提交事务
        $pdo->commit();
        $success = '视频删除成功';
        
    } catch (Exception $e) {
        // 回滚事务
        $pdo->rollBack();
        $error = "删除失败：" . $e->getMessage();
    }
}

// 在处理删除操作的代码后面添加批量操作的处理逻辑
// 处理批量删除操作
if (isset($_POST['action']) && $_POST['action'] == 'batch_delete') {
    if (isset($_POST['movie_ids']) && !empty($_POST['movie_ids'])) {
        $movie_ids = $_POST['movie_ids'];
        
        try {
            // 开启事务
            $pdo->beginTransaction();
            
            // 先删除相关留言
            $placeholders = implode(',', array_fill(0, count($movie_ids), '?'));
            $sql = "DELETE FROM comments WHERE movie_id IN ($placeholders)";
            $stmt = $pdo->prepare($sql);
            $stmt->execute($movie_ids);
            
            // 然后删除视频
            $sql = "DELETE FROM movies WHERE id IN ($placeholders)";
            $stmt = $pdo->prepare($sql);
            $stmt->execute($movie_ids);
            
            // 提交事务
            $pdo->commit();
            // 使用session存储成功消息
            $_SESSION['success_message'] = '批量删除成功';
            // 构建重定向URL并保留筛选参数
            $redirectUrl = 'movie_list.php';
            if (!empty($_POST['type'])) {
                $redirectUrl .= '?type=' . intval($_POST['type']);
                if (!empty($_POST['search'])) {
                    $redirectUrl .= '&search=' . urlencode($_POST['search']);
                }
            } elseif (!empty($_GET['type'])) {
                $redirectUrl .= '?type=' . intval($_GET['type']);
                if (!empty($_GET['search'])) {
                    $redirectUrl .= '&search=' . urlencode($_GET['search']);
                }
            }
            header("Location: $redirectUrl");
            exit;
        } catch (Exception $e) {
            // 回滚事务
            $pdo->rollBack();
            $error = "批量删除失败：" . $e->getMessage();
        }
    }
}

// 处理批量更改状态操作
if (isset($_POST['action']) && $_POST['action'] == 'batch_change_status') {
    if (isset($_POST['movie_ids']) && !empty($_POST['movie_ids']) && isset($_POST['status'])) {
        $movie_ids = $_POST['movie_ids'];
        $status = intval($_POST['status']);
        
        try {
            $placeholders = implode(',', array_fill(0, count($movie_ids), '?'));
            $sql = "UPDATE movies SET status = ? WHERE id IN ($placeholders)";
            $stmt = $pdo->prepare($sql);
            // 构造参数数组，第一个是状态，后面是电影ID
            $params = array_merge([$status], $movie_ids);
            $stmt->execute($params);
            
            $success = '批量更改状态成功';
            // 使用session存储成功消息
            $_SESSION['success_message'] = '批量更改状态成功';
            // 重定向到不带任何参数的页面
            $redirectUrl = 'movie_list.php';
            if (!empty($_POST['type'])) {
                $redirectUrl .= '?type=' . intval($_POST['type']);
                if (!empty($_POST['search'])) {
                    $redirectUrl .= '&search=' . urlencode($_POST['search']);
                }
            } elseif (!empty($_GET['type'])) {
                $redirectUrl .= '?type=' . intval($_GET['type']);
                if (!empty($_GET['search'])) {
                    $redirectUrl .= '&search=' . urlencode($_GET['search']);
                }
            }
            header("Location: $redirectUrl");
            exit;
        } catch (Exception $e) {
            $error = "批量更改状态失败：" . $e->getMessage();
        }
    }
}

// 删除这行代码
// 检查URL参数中的成功消息
// if (isset($_GET['success'])) {
//     $success = $_GET['success'];
// }

// 获取类型参数，默认为全部
$type = isset($_GET['type']) ? intval($_GET['type']) : 0;

// 在文件顶部移动分页参数的初始化，确保在任何重定向之前都已设置
// 获取分页参数
$page = isset($_GET['page']) ? intval($_GET['page']) : 1;
// 确保页码至少为1
$page = max(1, $page);
$pageSize = 20;
$offset = ($page - 1) * $pageSize;

// 获取类型参数，默认为全部
$type = isset($_GET['type']) ? intval($_GET['type']) : 0;

// 添加搜索参数处理
$search = isset($_GET['search']) ? trim($_GET['search']) : '';
$searchForSql = $search; // 保留原始搜索字符串

// 添加状态参数处理
$status = isset($_GET['status']) ? intval($_GET['status']) : -1; // -1表示全部状态

// 构建SQL查询
$sql = "SELECT m.*, c.name as category_name FROM movies m LEFT JOIN movie_categories c ON m.category_id = c.id WHERE 1=1";
if ($type > 0) {
    $sql .= " AND m.type = $type";
}
// 添加搜索条件
if (!empty($search)) {
    // 只在SQL查询中使用经过处理的版本
    $searchForSql = $pdo->quote('%' . $search . '%');
    $sql .= " AND (m.title LIKE $searchForSql OR m.subtitle LIKE $searchForSql OR m.director LIKE $searchForSql OR m.actors LIKE $searchForSql)";
}
// 添加状态筛选条件
if ($status >= 0) {
    $sql .= " AND m.status = $status";
}

// 获取总记录数
$totalCount = $pdo->query(str_replace('m.*, c.name as category_name', 'COUNT(*) AS count', $sql))->fetchColumn();
$totalPages = ceil($totalCount / $pageSize);

// 添加排序和分页
$sql .= " ORDER BY m.create_time DESC LIMIT $offset, $pageSize";

// 获取电影列表
$movies = $pdo->query($sql)->fetchAll();
?>
<!DOCTYPE html>
<html lang="zh-CN">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>视频列表 - 爱影视</title>
    <link rel="stylesheet" href="css/style.css">
</head>
<body>
    <?php include 'includes/header.php'; ?>
    
    <div class="container">
        <div class="content-wrapper">
            <?php include 'includes/sidebar.php'; ?>
            
            <div class="content">
                <div class="main-content">
                    <div class="page-header">
                        <h2>视频列表</h2>
                        <div class="header-actions">
                            <!-- 添加搜索表单 -->
                            <form method="get" action="movie_list.php" class="search-form">
                                <input type="hidden" name="type" value="<?php echo $type; ?>">
                                <input type="hidden" name="status" value="<?php echo $status; ?>">
                                <!-- 在表单中显示原始搜索字符串，而不是处理过的 -->
                                <input type="text" name="search" placeholder="搜索标题、副标题、导演、演员..." value="<?php echo htmlspecialchars($search); ?>">
                                <button type="submit" class="btn-search">搜索</button>
                            </form>
                            
                            <?php
                            // 获取视频类型列表
                            $videoTypes = array();
                            $sql = "SELECT id, name FROM video_types WHERE status = 1 ORDER BY sort_order ASC";
                            $result = $pdo->query($sql);
                            if ($result && $result->rowCount() > 0) {
                                while ($row = $result->fetch(PDO::FETCH_ASSOC)) {
                                    $videoTypes[$row['id']] = $row['name'];
                                }
                            }
                            
                            // 构建onchange事件的JavaScript代码
                            $searchParam = !empty($search) ? '&search=' . urlencode($search) : '';
                            $statusParam = $status >= 0 ? '&status=' . $status : '';
                            ?>
                            <select onchange="location.href='movie_list.php?type='+this.value+'<?php echo $searchParam . $statusParam; ?>'">
                                <option value="0" <?php echo $type == 0 ? 'selected' : ''; ?>>全部类型</option>
                                <?php foreach ($videoTypes as $typeId => $typeName): ?>
                                    <option value="<?php echo $typeId; ?>" <?php echo $type == $typeId ? 'selected' : ''; ?>><?php echo $typeName; ?></option>
                                <?php endforeach; ?>
                            </select>
                            
                            <!-- 添加状态筛选下拉框 -->
                            <select onchange="location.href='movie_list.php?status='+this.value+'&type=<?php echo $type; ?><?php echo $searchParam; ?>'">
                                <option value="-1" <?php echo $status == -1 ? 'selected' : ''; ?>>全部状态</option>
                                <option value="1" <?php echo $status == 1 ? 'selected' : ''; ?>>正常</option>
                                <option value="0" <?php echo $status == 0 ? 'selected' : ''; ?>>下架</option>
                            </select>
                            <a href="movie_edit.php" class="btn-primary">添加视频</a>
                        </div>
                    </div>
                    
                    <?php if (!empty($error)): ?>
                        <div class="alert alert-error"><?php echo $error; ?></div>
                    <?php endif; ?>
                    
                    <?php if (!empty($success)): ?>
                        <div class="alert alert-success"><?php echo $success; ?></div>
                    <?php endif; ?>
                    
                    <form id="batch-form" method="post" action="movie_list.php">
                        <input type="hidden" name="action" id="batch-action">
                        <input type="hidden" name="status" id="batch-status-value">
                        <input type="hidden" name="type" value="<?php echo $type; ?>">
                        <input type="hidden" name="page" value="<?php echo $page; ?>">
                        <input type="hidden" name="search" value="<?php echo htmlspecialchars($search); ?>">
                        
                        <!-- 保留原有的内容区域，不修改content的子容器样式 -->
                        <table class="data-table">
                            <thead>
                                <tr>
                                    <th><input type="checkbox" id="select-all"></th>
                                    <th>ID</th>
                                    <th>标题</th>
                                    <th>类型</th>
                                    <th>分类</th>
                                    <th>上映年份</th>
                                    <th>观看次数</th>
                                    <th>状态</th>
                                    <th>创建时间</th>
                                    <th>操作</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if (empty($movies)): ?>
                                    <tr>
                                        <td colspan="10" class="no-data">暂无数据</td>
                                    </tr>
                                <?php else: ?>
                                    <?php foreach ($movies as $movie): ?>
                                        <tr>
                                            <td><input type="checkbox" name="movie_ids[]" class="movie-checkbox" value="<?php echo $movie['id']; ?>"></td>
                                            <td><?php echo $movie['id']; ?></td>
                                            <td><?php echo $movie['title']; ?></td>
                                            <td>
                                                <?php 
                                                    echo isset($videoTypes[$movie['type']]) ? $videoTypes[$movie['type']] : '未知类型';
                                                ?>
                                            </td>
                                            <td><?php echo $movie['category_name']; ?></td>
                                            <td><?php echo $movie['release_year']; ?></td>
                                            <td><?php echo $movie['view_count']; ?></td>
                                            <td><?php echo $movie['status'] == 1 ? '<span class="status-active">正常</span>' : '<span class="status-inactive">下架</span>'; ?></td>
                                            <td><?php echo $movie['create_time']; ?></td>
                                            <td>
                                                <a href="movie_edit.php?id=<?php echo $movie['id']; ?>&type=<?php echo $type; ?>&page=<?php echo $page; ?><?php echo !empty($search) ? '&search=' . urlencode($search) : ''; ?><?php echo $status >= 0 ? '&status=' . $status : ''; ?>" class="btn-edit">编辑</a>
                                                <a href="movie_list.php?action=delete&id=<?php echo $movie['id']; ?>&type=<?php echo $type; ?>&page=<?php echo $page; ?><?php echo !empty($search) ? '&search=' . urlencode($search) : ''; ?><?php echo $status >= 0 ? '&status=' . $status : ''; ?>" class="btn-delete" onclick="return confirm('确定要删除吗？')">删除</a>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </tbody>
                        </table>
                        
                        <!-- 在表格下方、分页上方添加批量操作容器 -->
                        <?php if (!empty($movies)): ?>
                        <div class="batch-actions-container">
                            <div class="batch-actions">
                                <div class="batch-info">
                                    <label><input type="checkbox" id="select-all-bottom"> 全选</label>
                                    <span>已选择 <span id="selected-count">0</span> 项</span>
                                </div>
                                <div class="batch-controls">
                                    <select id="batch-status-select">
                                        <option value="">-- 更改状态 --</option>
                                        <option value="1">设为正常</option>
                                        <option value="0">设为下架</option>
                                    </select>
                                    <button id="batch-status-btn" class="btn-edit">批量更改状态</button>
                                    <button id="batch-delete-btn" class="btn-delete">批量删除</button>
                                </div>
                            </div>
                        </div>
                        <script>
                            // 获取两个全选框元素
                            const selectAllTop = document.getElementById('select-all');
                            const selectAllBottom = document.getElementById('select-all-bottom');
                            
                            // 全选/取消全选功能 - 顶部全选框
                            selectAllTop.addEventListener('change', function() {
                                updateCheckboxes(this.checked);
                            });
                            
                            // 全选/取消全选功能 - 底部全选框
                            selectAllBottom.addEventListener('change', function() {
                                updateCheckboxes(this.checked);
                            });
                            
                            // 统一的更新复选框函数
                            function updateCheckboxes(checked) {
                                const checkboxes = document.querySelectorAll('.movie-checkbox');
                                checkboxes.forEach(checkbox => {
                                    checkbox.checked = checked;
                                });
                                // 确保两个全选框状态同步
                                selectAllTop.checked = checked;
                                selectAllBottom.checked = checked;
                                updateSelectedCount();
                            }
                            
                            // 单个复选框更改事件
                            document.querySelectorAll('.movie-checkbox').forEach(checkbox => {
                                checkbox.addEventListener('change', function() {
                                    updateSelectedCount();
                                    // 检查是否所有复选框都被选中，同步更新全选框状态
                                    const allCheckboxes = document.querySelectorAll('.movie-checkbox');
                                    const allChecked = Array.from(allCheckboxes).every(cb => cb.checked);
                                    const anyChecked = Array.from(allCheckboxes).some(cb => cb.checked);
                                    
                                    selectAllTop.checked = allChecked;
                                    selectAllBottom.checked = allChecked;
                                    
                                    // 设置indeterminate状态（部分选中）
                                    selectAllTop.indeterminate = anyChecked && !allChecked;
                                    selectAllBottom.indeterminate = anyChecked && !allChecked;
                                });
                            });
                            
                            // 更新选中数量显示
                            function updateSelectedCount() {
                                const selectedCount = document.querySelectorAll('.movie-checkbox:checked').length;
                                document.getElementById('selected-count').textContent = selectedCount;
                            }
                            
                            // 批量更改状态
                            document.getElementById('batch-status-btn').addEventListener('click', function() {
                                const selectedStatus = document.getElementById('batch-status-select').value;
                                const selectedMovies = getSelectedMovies();
                                
                                if (selectedMovies.length === 0) {
                                    alert('请至少选择一个视频');
                                    return;
                                }
                                
                                if (!selectedStatus) {
                                    alert('请选择要更改的状态');
                                    return;
                                }
                                
                                document.getElementById('batch-action').value = 'batch_change_status';
                                document.getElementById('batch-status-value').value = selectedStatus;
                                document.getElementById('batch-form').submit();
                            });
                            
                            // 批量删除
                            document.getElementById('batch-delete-btn').addEventListener('click', function() {
                                const selectedMovies = getSelectedMovies();
                                
                                if (selectedMovies.length === 0) {
                                    alert('请至少选择一个视频');
                                    return;
                                }
                                
                                if (confirm('确定要删除选中的' + selectedMovies.length + '个视频吗？此操作不可恢复，并且会删除相关的评论。')) {
                                    document.getElementById('batch-action').value = 'batch_delete';
                                    document.getElementById('batch-form').submit();
                                }
                            });
                            
                            // 获取选中的电影ID
                            function getSelectedMovies() {
                                const checkboxes = document.querySelectorAll('.movie-checkbox:checked');
                                const movieIds = Array.from(checkboxes).map(checkbox => checkbox.value);
                                return movieIds;
                            }
                            
                            // 初始化选中数量显示
                            window.onload = function() {
                                updateSelectedCount();
                            };
                        </script>
                        <?php endif; ?>
                    </form>
                    
                    <!-- 分页 -->
                    <?php if ($totalPages > 1): ?>
                        <div class="pagination">
                            <a href="movie_list.php?type=<?php echo $type; ?>&page=1<?php echo !empty($search) ? '&search=' . urlencode($search) : ''; ?><?php echo $status >= 0 ? '&status=' . $status : ''; ?>" <?php echo $page == 1 ? 'class="disabled"' : ''; ?>>首页</a>
                            <a href="movie_list.php?type=<?php echo $type; ?>&page=<?php echo $page - 1; ?><?php echo !empty($search) ? '&search=' . urlencode($search) : ''; ?><?php echo $status >= 0 ? '&status=' . $status : ''; ?>" <?php echo $page == 1 ? 'class="disabled"' : ''; ?>>上一页</a>
                            
                            <?php for ($i = max(1, $page - 2); $i <= min($totalPages, $page + 2); $i++): ?>
                                <a href="movie_list.php?type=<?php echo $type; ?>&page=<?php echo $i; ?><?php echo !empty($search) ? '&search=' . urlencode($search) : ''; ?><?php echo $status >= 0 ? '&status=' . $status : ''; ?>" <?php echo $page == $i ? 'class="active"' : ''; ?>><?php echo $i; ?></a>
                            <?php endfor; ?>
                            
                            <a href="movie_list.php?type=<?php echo $type; ?>&page=<?php echo $page + 1; ?><?php echo !empty($search) ? '&search=' . urlencode($search) : ''; ?><?php echo $status >= 0 ? '&status=' . $status : ''; ?>" <?php echo $page == $totalPages ? 'class="disabled"' : ''; ?>>下一页</a>
                            <a href="movie_list.php?type=<?php echo $type; ?>&page=<?php echo $totalPages; ?><?php echo !empty($search) ? '&search=' . urlencode($search) : ''; ?><?php echo $status >= 0 ? '&status=' . $status : ''; ?>" <?php echo $page == $totalPages ? 'class="disabled"' : ''; ?>>末页</a>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
    
    <?php include 'includes/footer.php'; ?>
</body>
</html>