<?php
// 搜索采集功能页面
require_once 'includes/check_login.php';
require_once '../config.php';

// 数据库连接
$pdo = getDbConnection();

// 初始化变量
$error = '';
$success = '';
$totalProcessed = 0;
$totalSuccess = 0;
$totalFailed = 0;
$totalSkipped = 0; // 记录跳过的重复数据数量
// 添加分页相关变量
$currentPage = 1;
$hasMorePages = true;

// 添加类型名称映射数组（移到全局作用域）
$typeNameMapping = array(
    1 => '电影',
    2 => '剧集',
    3 => '动漫',
    4 => '综艺',
    5 => '纪录片',
    6 => '其他类型'
);

/// 处理JSON数据格式化和提取
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        if (isset($_POST['format_json'])) {
            // 获取粘贴的JSON数据
            $jsonContent = isset($_POST['json_content']) ? trim($_POST['json_content']) : '';
            
            if (empty($jsonContent)) {
                throw new Exception('请输入JSON数据');
            }
            
            // 修复：清理JSON数据中的控制字符
            $jsonContent = preg_replace('/[\x00-\x1F\x7F]/u', '', $jsonContent);
            
            // 解析JSON
            $data = json_decode($jsonContent, true);
            
            if (json_last_error() !== JSON_ERROR_NONE) {
                throw new Exception('JSON解析失败: ' . json_last_error_msg());
            }
            
            // 检查数据格式，支持两种格式：直接是电影数组或包含list键的对象
            if (!is_array($data)) {
                throw new Exception('JSON数据必须是数组格式');
            }
            
            // 如果数据包含list键，则使用list键的值
            if (isset($data['list']) && is_array($data['list'])) {
                $moviesData = $data['list'];
            } else {
                $moviesData = $data;
            }
            
            if (empty($moviesData)) {
                throw new Exception('JSON中没有找到电影数据');
            }
            
            // 保存提取的电影信息到session，以便后续处理
            $_SESSION['movies_data'] = $moviesData;
            
            // 格式化JSON并显示结果
            $formattedJson = json_encode($moviesData, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
            
            // 跳转到结果页面或直接显示结果
            $success = 'JSON数据格式化成功，共找到 ' . count($moviesData) . ' 部影片';
            
        } else if (isset($_POST['collect_selected'])) {
            // 处理用户选择的影片采集
            if (!isset($_SESSION['movies_data']) || !isset($_POST['selected_movies'])) {
                throw new Exception('没有选择要采集的影片数据');
            }
            
            $moviesData = $_SESSION['movies_data'];
            $selectedMovieIds = $_POST['selected_movies'];
            
            // 筛选出用户选择的影片
            $selectedMovies = array();
            foreach ($selectedMovieIds as $index) {
                if (isset($moviesData[$index])) {
                    $selectedMovies[] = $moviesData[$index];
                }
            }
            
            if (empty($selectedMovies)) {
                throw new Exception('请至少选择一部影片进行采集');
            }
            
            // 扩展类型映射数组，确保包含所有可能的映射关系
            $typeIdMapping = array(
                1 => 1,    // 电影
                2 => 2,    // 剧集
                3 => 4,    // 综艺
                4 => 3,    // 动漫
                5 => 5,    // 纪录片
                6 => 6,    // 其他类型
            );
            
            $categoryIdMapping = array(
                6 => 1,
                7 => 2,
                8 => 3,
                9 => 4,
                10 => 5,
                11 => 6,
                12 => 7,
                20 => 8,
                34 => 9,
                37 => 10,
                13 => 11,
                14 => 12,
                15 => 13,
                16 => 14,
                21 => 15,
                22 => 16,
                23 => 17,
                24 => 18,
                36 => 19,
                29 => 20,
                30 => 21,
                31 => 22,
                32 => 23,
                33 => 24,
                25 => 25,
                26 => 26,
                27 => 27,
                28 => 28
            );
            
            // 重置统计变量
            $totalProcessed = 0;
            $totalSuccess = 0;
            $totalFailed = 0;
            $totalSkipped = 0;
            
            // 开始事务
            $pdo->beginTransaction();
            
            foreach ($selectedMovies as $movieData) {
                $totalProcessed++;
                
                try {
                    // 检查是否已存在相同的电影
                    $existingMovieId = checkExistingMovie($movieData, $pdo, $typeIdMapping, $categoryIdMapping);
                    
                    if ($existingMovieId) {
                        // 电影已存在，增加跳过计数
                        $totalSkipped++;
                        
                        // 更新数据
                        $movieId = updateExistingMovie($movieData, $existingMovieId, $pdo, $typeIdMapping, $categoryIdMapping);
                        
                        // 处理剧集数据
                        if (isset($movieData['vod_play_url']) && !empty($movieData['vod_play_url'])) {
                            // 尝试处理剧集数据
                            $newEpisodesAdded = 0;
                            try {
                                processEpisodeData($movieData, $existingMovieId, $pdo, $newEpisodesAdded);
                            } catch (Exception $e) {
                                // 如果处理分集失败，记录错误但继续处理
                                error_log('处理分集数据失败: ' . $e->getMessage());
                            }
                            
                            // 如果添加了新剧集，则认为更新成功
                            if ($newEpisodesAdded > 0) {
                                $totalSuccess++;
                                // 移除跳过计数，因为更新了新剧集
                                $totalSkipped--;
                            }
                        }
                        
                    } else {
                        // 电影不存在，插入新数据
                        $movieId = processMovieData($movieData, $pdo, $typeIdMapping, $categoryIdMapping);
                        
                        // 处理剧集数据
                        if (isset($movieData['vod_play_url']) && !empty($movieData['vod_play_url'])) {
                            $newEpisodesAdded = 0;
                            // 尝试处理分集数据
                            try {
                                processEpisodeData($movieData, $movieId, $pdo, $newEpisodesAdded);
                            } catch (Exception $e) {
                                // 如果处理分集失败，记录错误但继续处理
                                error_log('处理分集数据失败: ' . $e->getMessage());
                            }
                        }
                        
                        $totalSuccess++;
                    }
                } catch (Exception $e) {
                    // 记录错误但继续处理下一个
                    error_log('处理电影数据失败: ' . $e->getMessage());
                    $totalFailed++;
                    continue;
                }
            }
            
            // 提交事务
            $pdo->commit();
            
            $success = "影片采集完成！总共处理 {$totalProcessed} 个视频，成功 {$totalSuccess} 个，失败 {$totalFailed} 个，跳过 {$totalSkipped} 个重复项";
            
            // 清除session中的数据
            unset($_SESSION['movies_data']);
            
        } else if (isset($_POST['clear_movies'])) {
            // 处理清除影片列表请求
            if (isset($_SESSION['movies_data'])) {
                unset($_SESSION['movies_data']);
                unset($_SESSION['search_keyword']);
                unset($_SESSION['current_page']);
                $success = '影片列表已成功清除';
            }
        } else if (isset($_POST['search_direct'])) {
            // 处理直接搜索请求
            $keyword = isset($_POST['keyword']) ? trim($_POST['keyword']) : '';
            if (empty($keyword)) {
                throw new Exception('请输入搜索关键词');
            }
            
            // 获取当前页码，默认为1
            $currentPage = isset($_POST['page']) ? intval($_POST['page']) : 1;
            
            // 构建搜索URL，包含页码参数
            $searchUrl = 'https://caiji.dyttzyapi.com/api.php/provide/vod/?ac=detail&wd=' . urlencode($keyword) . '&pg=' . $currentPage;
            
            // 使用cURL获取JSON数据
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $searchUrl);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
            curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
            curl_setopt($ch, CURLOPT_TIMEOUT, 30);
            
            // 添加更完整的浏览器模拟头部信息，避免被WAF拦截
            curl_setopt($ch, CURLOPT_HTTPHEADER, array(
                'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,image/webp,image/apng,*/*;q=0.8',
                'Accept-Language: zh-CN,zh;q=0.9,en;q=0.8',
                'Cache-Control: max-age=0',
                'Connection: keep-alive',
                'Host: caiji.dyttzyapi.com',
                'Upgrade-Insecure-Requests: 1',
                'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36'
            ));
            
            // 启用cookie支持
            curl_setopt($ch, CURLOPT_COOKIEJAR, tempnam(sys_get_temp_dir(), 'cookie'));
            curl_setopt($ch, CURLOPT_COOKIEFILE, tempnam(sys_get_temp_dir(), 'cookie'));
            
            // 允许重定向
            curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
            curl_setopt($ch, CURLOPT_MAXREDIRS, 3);
            
            $response = curl_exec($ch);
            $curlError = curl_error($ch);
            curl_close($ch);
            
            if ($curlError) {
                throw new Exception('搜索请求失败: ' . $curlError);
            }
            
            if (empty($response)) {
                throw new Exception('未获取到搜索结果');
            }
            
            // 详细记录原始响应的前200个字符用于调试
            $debugInfo = '原始响应前200字符: ' . substr($response, 0, 200);
            
            // 检查是否返回了403错误或其他非JSON内容
            if (strpos($response, '403 Forbidden') !== false || strpos($response, 'WAF') !== false) {
                throw new Exception('API访问被拒绝(403 Forbidden)，可能是WAF拦截了请求\n' . $debugInfo);
            }
            
            // 增强的JSON数据清理，移除所有非UTF-8字符和控制字符
            $response = preg_replace('/[\x00-\x1F\x7F-\x9F]/u', '', $response); // 移除控制字符
            $response = preg_replace('/\s+/', ' ', $response); // 将多余空白替换为单个空格
            $response = trim($response); // 去除首尾空白
            
            // 尝试在错误情况下找到有效的JSON部分
            if (strpos($response, '{') !== false && strrpos($response, '}') !== false) {
                $startPos = strpos($response, '{');
                $endPos = strrpos($response, '}') + 1;
                $response = substr($response, $startPos, $endPos - $startPos);
            } else if (strpos($response, '[') !== false && strrpos($response, ']') !== false) {
                $startPos = strpos($response, '[');
                $endPos = strrpos($response, ']') + 1;
                $response = substr($response, $startPos, $endPos - $startPos);
            }
            
            // 解析JSON
            $data = json_decode($response, true);
            
            if (json_last_error() !== JSON_ERROR_NONE) {
                throw new Exception('JSON解析失败: ' . json_last_error_msg() . '\n' . $debugInfo);
            }
            
            // 检查数据格式，支持两种格式：直接是电影数组或包含list键的对象
            if (!is_array($data)) {
                throw new Exception('JSON数据必须是数组格式\n' . $debugInfo);
            }
            
            // 如果数据包含list键，则使用list键的值
            if (isset($data['list']) && is_array($data['list'])) {
                $moviesData = $data['list'];
            } else {
                $moviesData = $data;
            }
            
            if (empty($moviesData)) {
                // 对于非第一页的请求，如果没有数据，我们需要特殊处理
                if ($currentPage > 1) {
                    // 保存空结果到session，这样前端可以显示"没有更多内容"
                    $_SESSION['movies_data'] = array();
                    $_SESSION['search_keyword'] = $keyword;
                    $_SESSION['current_page'] = $currentPage;
                    $_SESSION['has_more_pages'] = false;
                    $success = '已显示全部内容，没有更多影片';
                } else {
                    throw new Exception('未找到相关影片数据');
                }
            } else {
                // 保存提取的电影信息和搜索参数到session
                $_SESSION['movies_data'] = $moviesData;
                $_SESSION['search_keyword'] = $keyword;
                $_SESSION['current_page'] = $currentPage;
                
                // 默认假设还有下一页
                $_SESSION['has_more_pages'] = true;
                
                // 如果当前不是第一页，我们可以根据当前页结果数量来判断是否有下一页
                if ($currentPage > 1 && count($moviesData) < 20) { // 假设每页至少返回20条数据
                    $_SESSION['has_more_pages'] = false;
                }
                
                $success = '搜索成功，共找到 ' . count($moviesData) . ' 部影片';
            }
        } else if (isset($_POST['clear_movies'])) {
            // 处理清除影片列表请求
            if (isset($_SESSION['movies_data'])) {
                unset($_SESSION['movies_data']);
                unset($_SESSION['search_keyword']);
                unset($_SESSION['current_page']);
                $success = '影片列表已成功清除';
            }
        }
    } catch (Exception $e) {
        // 回滚事务
        if (isset($pdo) && $pdo->inTransaction()) {
            $pdo->rollBack();
        }
        
        $error = '操作失败：' . $e->getMessage();
    }
}

/**
 * 检查是否存在相同的电影
 */
function checkExistingMovie($movieData, $pdo, $typeIdMapping, $categoryIdMapping) {
    // 映射数据
    $type = 1; // 默认类型
    if (isset($movieData['type_id_1']) && isset($typeIdMapping[$movieData['type_id_1']])) {
        $type = $typeIdMapping[$movieData['type_id_1']];
    }
    
    $title = isset($movieData['vod_name']) ? $movieData['vod_name'] : '';
    
    if (empty($title)) {
        throw new Exception('视频标题不能为空');
    }
    
    // 检查数据库中是否已存在相同标题和类型的电影
    $sql = "SELECT id FROM movies WHERE title = ? AND type = ? LIMIT 1";
    $stmt = $pdo->prepare($sql);
    $stmt->execute([$title, $type]);
    $result = $stmt->fetch(PDO::FETCH_ASSOC);
    
    return $result ? $result['id'] : null;
}

/**
 * 计算播放链接中的实际地址数量
 */
function calculateActualEpisodeCount($vodPlayUrl) {
    if (!isset($vodPlayUrl) || empty($vodPlayUrl)) {
        return 1; // 如果没有播放链接，默认为1集
    }
    
    // 提取$$$后面的内容
    $parts = explode('$$$', $vodPlayUrl);
    $urlContent = count($parts) > 1 ? $parts[1] : $vodPlayUrl;
    
    // 使用parseEpisodes函数来获取实际的剧集数量
    $episodes = parseEpisodes($urlContent);
    
    return !empty($episodes) ? count($episodes) : 1;
}

/**
 * 更新已存在的电影数据
 */
function updateExistingMovie($movieData, $movieId, $pdo, $typeIdMapping, $categoryIdMapping) {
    // 映射数据
    $type = 1; // 默认类型
    if (isset($movieData['type_id_1']) && isset($typeIdMapping[$movieData['type_id_1']])) {
        $type = $typeIdMapping[$movieData['type_id_1']];
    }
    
    $categoryId = 1; // 默认分类
    if (isset($movieData['type_id']) && isset($categoryIdMapping[$movieData['type_id']])) {
        $categoryId = $categoryIdMapping[$movieData['type_id']];
    }
    
    // 处理播放链接
    $playUrl = '';
    $totalEpisodes = isset($movieData['vod_total']) ? intval($movieData['vod_total']) : 1;
    
    // 如果vod_total为0，则计算实际的剧集数量
    if ($totalEpisodes === 0 && isset($movieData['vod_play_url'])) {
        $totalEpisodes = calculateActualEpisodeCount($movieData['vod_play_url']);
    }
    
    // 始终尝试提取第一个可播放URL作为主播放链接
    if (isset($movieData['vod_play_url']) && !empty($movieData['vod_play_url'])) {
        // 提取$$$后面的内容
        $parts = explode('$$$', $movieData['vod_play_url']);
        if (count($parts) > 1) {
            $urlContent = $parts[1];
            
            // 处理链接内容
            $playUrl = extractFirstPlayableUrl($urlContent);
        } else {
            // 如果没有$$$，则直接处理整个链接
            $playUrl = extractFirstPlayableUrl($movieData['vod_play_url']);
        }
    }
    
    // 准备更新数据
    $title = isset($movieData['vod_name']) ? $movieData['vod_name'] : '';
    $subtitle = '';
    if (isset($movieData['vod_sub'])) {
        $subtitle = $movieData['vod_sub'];
    } elseif (isset($movieData['vod_en'])) {
        $subtitle = $movieData['vod_en'];
    }
    
    $director = isset($movieData['vod_director']) ? $movieData['vod_director'] : '';
    $actors = isset($movieData['vod_actor']) ? $movieData['vod_actor'] : '';
    $region = isset($movieData['vod_area']) ? $movieData['vod_area'] : '';
    $language = isset($movieData['vod_lang']) ? $movieData['vod_lang'] : '';
    $releaseYear = isset($movieData['vod_year']) ? $movieData['vod_year'] : '';
    $coverImg = isset($movieData['vod_pic']) ? $movieData['vod_pic'] : '';
    $description = isset($movieData['vod_content']) ? $movieData['vod_content'] : '';
    
    // 更新电影数据
    $sql = "UPDATE movies SET title = ?, subtitle = ?, type = ?, category_id = ?, director = ?, actors = ?, region = ?, language = ?, release_year = ?, cover_img = ?, description = ?, play_url = ?, total_episodes = ? WHERE id = ?";
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute([$title, $subtitle, $type, $categoryId, $director, $actors, $region, $language, $releaseYear, $coverImg, $description, $playUrl, $totalEpisodes, $movieId]);
    
    return $movieId;
}

/**
 * 处理电影数据并保存到数据库
 */
function processMovieData($movieData, $pdo, $typeIdMapping, $categoryIdMapping) {
    // 映射数据
    $type = 1; // 默认类型
    if (isset($movieData['type_id_1']) && isset($typeIdMapping[$movieData['type_id_1']])) {
        $type = $typeIdMapping[$movieData['type_id_1']];
    }
    
    $categoryId = 1; // 默认分类
    if (isset($movieData['type_id']) && isset($categoryIdMapping[$movieData['type_id']])) {
        $categoryId = $categoryIdMapping[$movieData['type_id']];
    }
    
    // 处理播放链接
    $playUrl = '';
    $totalEpisodes = isset($movieData['vod_total']) ? intval($movieData['vod_total']) : 1;
    
    // 如果vod_total为0，则计算实际的剧集数量
    if ($totalEpisodes === 0 && isset($movieData['vod_play_url'])) {
        $totalEpisodes = calculateActualEpisodeCount($movieData['vod_play_url']);
    }
    
    // 始终尝试提取第一个可播放URL作为主播放链接
    if (isset($movieData['vod_play_url']) && !empty($movieData['vod_play_url'])) {
        // 提取$$$后面的内容
        $parts = explode('$$$', $movieData['vod_play_url']);
        if (count($parts) > 1) {
            $urlContent = $parts[1];
            
            // 处理链接内容
            $playUrl = extractFirstPlayableUrl($urlContent);
        } else {
            // 如果没有$$$，则直接处理整个链接
            $playUrl = extractFirstPlayableUrl($movieData['vod_play_url']);
        }
    }
    
    // 准备插入数据
    $title = isset($movieData['vod_name']) ? $movieData['vod_name'] : '';
    $subtitle = '';
    if (isset($movieData['vod_sub'])) {
        $subtitle = $movieData['vod_sub'];
    } elseif (isset($movieData['vod_en'])) {
        $subtitle = $movieData['vod_en'];
    }
    
    $director = isset($movieData['vod_director']) ? $movieData['vod_director'] : '';
    $actors = isset($movieData['vod_actor']) ? $movieData['vod_actor'] : '';
    $region = isset($movieData['vod_area']) ? $movieData['vod_area'] : '';
    $language = isset($movieData['vod_lang']) ? $movieData['vod_lang'] : '';
    $releaseYear = isset($movieData['vod_year']) ? $movieData['vod_year'] : '';
    $coverImg = isset($movieData['vod_pic']) ? $movieData['vod_pic'] : '';
    $description = isset($movieData['vod_content']) ? $movieData['vod_content'] : '';
    
    // 验证必填字段
    if (empty($title)) {
        throw new Exception('视频标题不能为空');
    }
    
    // 插入电影数据
    $sql = "INSERT INTO movies (title, subtitle, type, category_id, director, actors, region, language, release_year, cover_img, description, play_url, total_episodes, status, create_time) 
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())";
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute([$title, $subtitle, $type, $categoryId, $director, $actors, $region, $language, $releaseYear, $coverImg, $description, $playUrl, $totalEpisodes, 1]);
    
    // 返回插入的电影ID
    return $pdo->lastInsertId();
}

/**
 * 处理剧集数据并保存到数据库
 */
function processEpisodeData($movieData, $movieId, $pdo, &$newEpisodesAdded) {
    $newEpisodesAdded = 0; // 初始化新增剧集计数
    
    if (!isset($movieData['vod_play_url']) || empty($movieData['vod_play_url'])) {
        throw new Exception('vod_play_url不存在或为空');
    }
    
    // 提取$$$后面的内容
    $parts = explode('$$$', $movieData['vod_play_url']);
    $urlContent = count($parts) > 1 ? $parts[1] : $movieData['vod_play_url'];
    
    // 解析剧集数据
    $episodes = parseEpisodes($urlContent);
    
    if (empty($episodes)) {
        throw new Exception('未找到有效的剧集数据');
    }
    
    // 插入分集数据
    foreach ($episodes as $episodeNum => $episodeData) {
        // 跳过空链接
        if (empty(trim($episodeData['url']))) {
            continue;
        }
        
        // 替换m3u8链接
        $playUrl = str_replace('index.m3u8', '3000k/hls/mixed.m3u8', $episodeData['url']);
        
        // 检查是否已存在相同的剧集
        if (checkExistingEpisode($movieId, $episodeNum, $pdo)) {
            continue; // 跳过已存在的剧集
        }
        
        // 准备SQL语句
        if (!empty($episodeData['title'])) {
            $sql = "INSERT INTO episodes (movie_id, episode_num, episode_title, play_url, create_time) VALUES (?, ?, ?, ?, NOW())";
            $stmt = $pdo->prepare($sql);
            $stmt->execute([$movieId, $episodeNum, $episodeData['title'], $playUrl]);
        } else {
            $sql = "INSERT INTO episodes (movie_id, episode_num, play_url, create_time) VALUES (?, ?, ?, NOW())";
            $stmt = $pdo->prepare($sql);
            $stmt->execute([$movieId, $episodeNum, $playUrl]);
        }
        
        $newEpisodesAdded++; // 增加新增剧集计数
    }
}

/**
 * 检查是否存在相同的剧集
 */
function checkExistingEpisode($movieId, $episodeNum, $pdo) {
    $sql = "SELECT id FROM episodes WHERE movie_id = ? AND episode_num = ? LIMIT 1";
    $stmt = $pdo->prepare($sql);
    $stmt->execute([$movieId, $episodeNum]);
    $result = $stmt->fetch(PDO::FETCH_ASSOC);
    
    return $result !== false;
}

/**
 * 解析剧集数据，支持#和$分隔符
 */
function parseEpisodes($urlContent) {
    $episodes = array();
    $episodeNum = 1;
    
    // 首先尝试用#分割（主要分隔符）
    $segments = explode('#', $urlContent);
    
    if (count($segments) > 1) {
        foreach ($segments as $segment) {
            if (empty(trim($segment))) {
                continue;
            }
            
            // 检查是否有$分隔的标题和链接
            if (strpos($segment, '$') !== false) {
                list($title, $url) = explode('$', $segment, 2);
                $episodes[$episodeNum++] = array(
                    'title' => trim($title),
                    'url' => trim($url)
                );
            } else {
                // 可能是单独的标题或链接
                $episodes[$episodeNum++] = array(
                    'title' => '',
                    'url' => trim($segment)
                );
            }
        }
    } else {
        // 如果#分割失败，尝试用$分割
        $segments = explode('$', $urlContent);
        
        for ($i = 0; $i < count($segments); $i += 2) {
            if (isset($segments[$i+1]) && !empty(trim($segments[$i+1]))) {
                $episodes[$episodeNum++] = array(
                    'title' => trim($segments[$i]),
                    'url' => trim($segments[$i+1])
                );
            }
        }
    }
    
    return $episodes;
}

/**
 * 提取第一个可播放的URL
 */
function extractFirstPlayableUrl($urlContent) {
    // 解析所有剧集
    $episodes = parseEpisodes($urlContent);
    
    // 返回第一个有效的URL
    foreach ($episodes as $episode) {
        if (!empty($episode['url'])) {
            // 替换m3u8链接
            return str_replace('index.m3u8', '3000k/hls/mixed.m3u8', $episode['url']);
        }
    }
    
    return '';
}
?>
<!DOCTYPE html>
<html lang="zh-CN">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>搜索采集 - 爱影视</title>
    <link rel="stylesheet" href="css/style.css">
    <script src="js/main.js"></script>
</head>
<body>
    <?php include 'includes/header.php'; ?>
    
    <div class="container">
        <div class="content-wrapper">
            <?php include 'includes/sidebar.php'; ?>
            
            <div class="content">
                <div class="main-content">
                    <div class="breadcrumb">
                        <a href="index.php">首页</a> &gt; 
                        <span>搜索采集</span>
                    </div>
                    
                    <div class="panel">
                        <div class="panel-header">
                            <h3>搜索采集</h3>
                        </div>
                        
                        <div class="panel-body">
                            <?php if ($error): ?>
                                <div class="alert alert-error"><?php echo $error; ?></div>
                            <?php endif; ?>
                            
                            <?php if ($success): ?>
                                <div class="alert alert-success"><?php echo $success; ?></div>
                            <?php endif; ?>
                            
                            <!-- JSON数据输入表单 -->
                            <div class="form-container">
                                
                                <!-- 搜索功能 -->
                                <div class="search-section">
                                    <h4>资源搜索</h4>
                                    <form id="searchForm" method="POST" action="" class="search-form">
                                        <input type="hidden" name="search_direct" value="1">
                                        <input type="text" id="searchKeyword" name="keyword" placeholder="请输入搜索关键词" class="form-input">
                                        <button type="submit" class="btn-search">搜索</button>
                                    </form>
                                    <p class="form-note">目前仅支持电影天堂资源网的JSON数据</p>
                                </div>
                                
                                <!-- 移除备用的JSON粘贴方式 -->
                                
                                <!-- 显示格式化的JSON数据和影片选择列表 -->
                                <?php if (isset($_SESSION['movies_data'])): ?>
                                    <div class="movies-list-section">
                                        <div class="section-header">
                                            <h4>影片列表 (<?php echo count($_SESSION['movies_data']); ?>) 
                                                <?php if (isset($_SESSION['current_page'])): ?>
                                                    <small>当前第 <?php echo $_SESSION['current_page']; ?> 页</small>
                                                <?php endif; ?>
                                            </h4>
                                            <button type="button" onclick="clearMovieList()" class="btn btn-danger">
                                                清除列表
                                            </button>
                                        </div>
                                        <form method="POST" action="">
                                            <div style="border: 1px solid #ddd; border-radius: 4px; padding: 15px; background-color: #2d3748;">
                                                <?php foreach ($_SESSION['movies_data'] as $index => $movie): ?>
                                                    <div style="display: flex; align-items: flex-start; padding: 10px; border-bottom: 1px solid #4a5568;">
                                                        <input type="checkbox" name="selected_movies[]" value="<?php echo $index; ?>" style="margin-right: 15px; margin-top: 5px;">
                                                        <div style="flex: 1;">
                                                            <h5 style="margin: 0 0 5px 0; color: #e0e0e0;"><?php echo isset($movie['vod_name']) ? $movie['vod_name'] : '未知标题'; ?></h5>
                                                            <div style="font-size: 14px; color: #cbd5e0;">
                                                                <?php if (isset($movie['type_id_1'])) echo '类型: ' . (isset($typeNameMapping[$movie['type_id_1']]) ? $typeNameMapping[$movie['type_id_1']] : $movie['type_id_1']) . ' | '; ?>
                                                                <?php if (isset($movie['vod_actor'])) echo '演员: ' . $movie['vod_actor'] . ' | '; ?>
                                                                <?php if (isset($movie['vod_area'])) echo '地区: ' . $movie['vod_area'] . ' | '; ?>
                                                                <?php if (isset($movie['vod_year'])) echo '年份: ' . $movie['vod_year']; ?>
                                                            </div>
                                                            <?php if (isset($movie['vod_pic'])): ?>
                                                                <img src="<?php echo $movie['vod_pic']; ?>" style="max-width: 120px; max-height: 160px; margin-top: 10px; border-radius: 4px;">
                                                            <?php endif; ?>
                                                        </div>
                                                    </div>
                                                <?php endforeach; ?>
                                            </div>
                                            <div style="margin-top: 20px; text-align: center;">
                                                <button type="button" onclick="selectAllMovies()" style="padding: 8px 15px; margin-right: 10px;">全选</button>
                                                <button type="button" onclick="deselectAllMovies()" style="padding: 8px 15px; margin-right: 30px;">取消全选</button>
                                                <button type="submit" name="collect_selected" class="btn-primary" style="padding: 10px 30px; font-size: 18px;">采集选中影片</button>
                                            </div>
                                        </form>
                                         
                                        <!-- 添加分页控制 -->
                                        <div style="margin-top: 20px; text-align: center;">
                                            <form id="paginationForm" method="POST" action="" style="display: inline-block;">
                                                <input type="hidden" name="search_direct" value="1">
                                                <input type="hidden" id="paginationKeyword" name="keyword" value="<?php echo isset($_SESSION['search_keyword']) ? $_SESSION['search_keyword'] : ''; ?>">
                                                <input type="hidden" id="paginationPage" name="page" value="<?php echo isset($_SESSION['current_page']) ? $_SESSION['current_page'] : 1; ?>">
                                                
                                                <?php if (isset($_SESSION['current_page']) && $_SESSION['current_page'] > 1): ?>
                                                    <button type="button" onclick="goToPage(<?php echo $_SESSION['current_page'] - 1; ?>)" style="padding: 8px 15px; margin-right: 10px;">上一页</button>
                                                <?php endif; ?>
                                                
                                                <!-- 显示当前页码 -->
                                                <span style="margin: 0 10px; padding: 8px 15px; background-color: #3182ce; color: white; border-radius: 4px;">
                                                    当前第 <?php echo isset($_SESSION['current_page']) ? $_SESSION['current_page'] : 1; ?> 页
                                                </span>
                                                
                                                <?php if (isset($_SESSION['has_more_pages']) && $_SESSION['has_more_pages'] === true): ?>
                                                    <button type="button" onclick="goToPage(<?php echo isset($_SESSION['current_page']) ? $_SESSION['current_page'] + 1 : 2; ?>)" style="padding: 8px 15px; margin-left: 10px;">下一页</button>
                                                <?php else: ?>
                                                    <?php if (isset($_SESSION['movies_data']) && count($_SESSION['movies_data']) > 0): ?>
                                                        <span style="color: #999; margin-left: 10px;">没有更多内容</span>
                                                    <?php endif; ?>
                                                <?php endif; ?>
                                            </form>
                                        </div>
                                    </div>
                                <?php endif; ?>

                                <!-- 添加搜索功能的JavaScript -->
                                <script>
                                // 按回车键触发搜索
                                document.getElementById('searchKeyword').addEventListener('keypress', function(e) {
                                    if (e.key === 'Enter') {
                                        document.getElementById('searchForm').submit();
                                    }
                                });
                                 
                                // 全选和取消全选功能
                                function selectAllMovies() {
                                    const checkboxes = document.querySelectorAll('input[name="selected_movies[]"]');
                                    checkboxes.forEach(checkbox => checkbox.checked = true);
                                }
                                 
                                function deselectAllMovies() {
                                    const checkboxes = document.querySelectorAll('input[name="selected_movies[]"]');
                                    checkboxes.forEach(checkbox => checkbox.checked = false);
                                }
                                 
                                // 清除临时列表功能
                                function clearMovieList() {
                                    if (confirm('确定要清除当前影片列表吗？')) {
                                        // 这里通过表单提交清除session中的数据
                                        const form = document.createElement('form');
                                        form.method = 'POST';
                                        form.style.display = 'none';
                                        const input = document.createElement('input');
                                        input.type = 'hidden';
                                        input.name = 'clear_movies';
                                        input.value = '1';
                                        form.appendChild(input);
                                        document.body.appendChild(form);
                                        form.submit();
                                    }
                                }
                                 
                                // 分页功能
                                function goToPage(pageNum) {
                                    // 移除旧的加载提示（如果存在）
                                    const oldLoadingIndicator = document.querySelector('.loading-indicator');
                                    if (oldLoadingIndicator) {
                                        oldLoadingIndicator.remove();
                                    }
                                    
                                    // 显示加载状态，使用和搜索成功提示相同的样式
                                    const loadingIndicator = document.createElement('div');
                                    loadingIndicator.className = 'loading-indicator alert alert-success';
                                    loadingIndicator.textContent = '加载中...';
                                    loadingIndicator.style.textAlign = 'center';
                                    loadingIndicator.style.margin = '20px 0';
                                    
                                    // 添加到页面顶部
                                    const panelBody = document.querySelector('.panel-body');
                                    panelBody.insertBefore(loadingIndicator, panelBody.firstChild);
                                    
                                    // 设置页码并提交表单
                                    document.getElementById('paginationPage').value = pageNum;
                                    document.getElementById('paginationForm').submit();
                                }
                                </script>
                                
                                <?php if ($totalProcessed > 0): ?>
                                    <div style="margin-top: 20px; text-align: left;">
                                        <h4>导入统计</h4>
                                        <ul>
                                            <li>总处理数: <?php echo $totalProcessed; ?></li>
                                            <li>成功数: <?php echo $totalSuccess; ?></li>
                                            <li>失败数: <?php echo $totalFailed; ?></li>
                                            <li>跳过重复: <?php echo $totalSkipped; ?></li>
                                        </ul>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <?php include 'includes/footer.php'; ?>
    
    <script>
     // 初始化视频分类数据
     const videoCategories = <?php echo isset($categories) ? json_encode($categories) : '[]'; ?>;
     </script>
</body>
</html>