<?php
// 视频采集功能页面
require_once 'includes/check_login.php';
require_once '../config.php';

// 数据库连接
$pdo = getDbConnection();

// 初始化变量
$error = '';
$success = '';
$totalProcessed = 0;
$totalSuccess = 0;
$totalFailed = 0;
$totalSkipped = 0; // 新增：记录跳过的重复数据数量

// 获取视频类型列表
$videoTypes = $pdo->query("SELECT id, name FROM video_types WHERE status = 1 ORDER BY sort_order ASC")->fetchAll();

// 获取所有分类列表并按类型分组
$categories = array();
// 修改前：包含了不存在的sort_order列
// $categoryResult = $pdo->query("SELECT id, name, type FROM movie_categories WHERE status = 1 ORDER BY type, sort_order ASC");
// 修改后：移除sort_order列，只按类型和ID排序
$categoryResult = $pdo->query("SELECT id, name, type FROM movie_categories WHERE status = 1 ORDER BY type, id ASC");
while ($cat = $categoryResult->fetch(PDO::FETCH_ASSOC)) {
    if (!isset($categories[$cat['type']])) {
        $categories[$cat['type']] = array();
    }
    $categories[$cat['type']][] = $cat;
}

// 处理一键采集和批量采集
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        // 获取选择的视频类型和分类
        $selectedType = isset($_POST['video_type']) ? intval($_POST['video_type']) : 0;
        $selectedCategory = isset($_POST['video_category']) ? intval($_POST['video_category']) : 0;
        $pageStart = isset($_POST['page_start']) ? intval($_POST['page_start']) : 1;
        $pageEnd = isset($_POST['page_end']) ? intval($_POST['page_end']) : 1;
        
        // 验证页码
        if ($pageStart < 1) $pageStart = 1;
        if ($pageEnd < $pageStart) $pageEnd = $pageStart;
        
        // 扩展类型映射数组，确保包含所有可能的映射关系
        $typeIdMapping = array(
            1 => 1,    // 电影
            2 => 2,    // 剧集
            3 => 4,    // 综艺
            4 => 3,    // 动漫
            5 => 5,    // 纪录片
            6 => 6,    // 其他类型
        );
        
        $categoryIdMapping = array(
            6 => 1,
            7 => 2,
            8 => 3,
            9 => 4,
            10 => 5,
            11 => 6,
            12 => 7,
            20 => 8,
            34 => 9,
            37 => 10,
            13 => 11,
            14 => 12,
            15 => 13,
            16 => 14,
            21 => 15,
            22 => 16,
            23 => 17,
            24 => 18,
            36 => 19,
            29 => 20,
            30 => 21,
            31 => 22,
            32 => 23,
            33 => 24,
            25 => 25,
            26 => 26,
            27 => 27,
            28 => 28
        );
        
        // 构建反向映射数组，用于根据用户选择的类型查找API类型ID
        $reverseTypeIdMapping = array_flip($typeIdMapping);
        $reverseCategoryIdMapping = array_flip($categoryIdMapping);
        
        // 设置超时时间（增加到60秒）
        ini_set('default_socket_timeout', 60);
        
        // 重置统计变量
        $totalProcessed = 0;
        $totalSuccess = 0;
        $totalFailed = 0;
        $totalSkipped = 0;
        $pagesSuccess = 0;
        $pagesFailed = 0;
        
        // 确定是单页采集还是批量采集
        $isBatchCollect = isset($_POST['batch_collect']);
        $startPage = $isBatchCollect ? $pageStart : $pageStart;
        $endPage = $isBatchCollect ? $pageEnd : $pageStart;
        
        // 循环处理每一页
        for ($pageNum = $startPage; $pageNum <= $endPage; $pageNum++) {
            try {
                // 采集地址 - 根据选择的类型构建不同的采集URL
                $collectUrl = 'http://caiji.dyttzyapi.com/api.php/provide/vod/?ac=detail';
                // 添加分页参数
                $collectUrl .= '&pg=' . $pageNum;
                
                if ($selectedType > 0) {
                    // 记录用户选择的类型
                    error_log('用户选择的视频类型ID: ' . $selectedType);
                    
                    // 直接使用typeIdMapping获取API类型ID
                    if (isset($reverseTypeIdMapping[$selectedType])) {
                        $apiTypeId = $reverseTypeIdMapping[$selectedType];
                        $collectUrl .= '&type_id_1=' . $apiTypeId;
                        error_log('映射到的API类型ID: ' . $apiTypeId);
                    }
                }
                
                // 如果选择了分类，添加到采集URL
                if ($selectedCategory > 0) {
                    if (isset($reverseCategoryIdMapping[$selectedCategory])) {
                        $apiCategoryId = $reverseCategoryIdMapping[$selectedCategory];
                        $collectUrl .= '&type_id=' . $apiCategoryId;
                        error_log('映射到的API分类ID: ' . $apiCategoryId);
                    }
                }
                
                // 设置HTTP请求头，模拟浏览器访问
                $options = [
                    'http' => [
                        'header' => "User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36\r\n"
                    ]
                ];
                
                $context = stream_context_create($options);
                
                // 获取JSON数据
                $jsonData = @file_get_contents($collectUrl, false, $context);
                
                if ($jsonData === false) {
                    // 获取最后一个错误
                    $errorInfo = error_get_last();
                    throw new Exception('无法连接到采集地址: ' . ($errorInfo ? $errorInfo['message'] : '未知错误'));
                }
                
                // 解析JSON
                $data = json_decode($jsonData, true);
                
                if (json_last_error() !== JSON_ERROR_NONE) {
                    throw new Exception('JSON解析失败: ' . json_last_error_msg());
                }
                
                // 检查code是否为1
                if (!isset($data['code']) || $data['code'] != 1) {
                    throw new Exception('采集数据格式错误，code不为1');
                }
                
                // 检查是否有list数据
                if (!isset($data['list']) || !is_array($data['list'])) {
                    throw new Exception('未找到list数据');
                }
                
                // 开始事务
                $pdo->beginTransaction();
                
                $pageProcessed = 0;
                $pageSuccess = 0;
                $pageFailed = 0;
                $pageSkipped = 0;
                
                foreach ($data['list'] as $movieData) {
                    // 如果用户选择了特定类型，检查当前电影的类型是否匹配
                    if ($selectedType > 0) {
                        // 获取当前电影的类型映射
                        $currentMovieType = 1; // 默认类型
                        if (isset($movieData['type_id_1']) && isset($typeIdMapping[$movieData['type_id_1']])) {
                            $currentMovieType = $typeIdMapping[$movieData['type_id_1']];
                        }
                        
                        // 如果类型不匹配，跳过此电影
                        if ($currentMovieType != $selectedType) {
                            continue;
                        }
                    }
                    
                    // 如果用户选择了特定分类，检查当前电影的分类是否匹配
                    if ($selectedCategory > 0) {
                        // 获取当前电影的分类映射
                        $currentMovieCategory = 1; // 默认分类
                        if (isset($movieData['type_id']) && isset($categoryIdMapping[$movieData['type_id']])) {
                            $currentMovieCategory = $categoryIdMapping[$movieData['type_id']];
                        }
                        
                        // 如果分类不匹配，跳过此电影
                        if ($currentMovieCategory != $selectedCategory) {
                            continue;
                        }
                    }
                    
                    $pageProcessed++;
                    $totalProcessed++;
                    
                    try {
                        // 检查是否已存在相同的电影
                        $existingMovieId = checkExistingMovie($movieData, $pdo, $typeIdMapping, $categoryIdMapping);
                        
                        if ($existingMovieId) {
                            // 电影已存在，增加跳过计数
                            $pageSkipped++;
                            $totalSkipped++;
                            
                            // 更新数据
                            $movieId = updateExistingMovie($movieData, $existingMovieId, $pdo, $typeIdMapping, $categoryIdMapping);
                            
                            // 处理剧集数据
                            if (isset($movieData['vod_play_url']) && !empty($movieData['vod_play_url'])) {
                                // 尝试处理剧集数据，但不再清空现有剧集
                                $newEpisodesAdded = 0;
                                try {
                                    processEpisodeData($movieData, $existingMovieId, $pdo, $newEpisodesAdded);
                                } catch (Exception $e) {
                                    // 如果处理分集失败，记录错误但继续处理
                                    error_log('处理分集数据失败: ' . $e->getMessage());
                                }
                                
                                // 如果添加了新剧集，则认为更新成功
                                if ($newEpisodesAdded > 0) {
                                    $pageSuccess++;
                                    $totalSuccess++;
                                    // 移除跳过计数，因为更新了新剧集
                                    $pageSkipped--;
                                    $totalSkipped--;
                                }
                            }
                            
                        } else {
                            // 电影不存在，插入新数据
                            $movieId = processMovieData($movieData, $pdo, $typeIdMapping, $categoryIdMapping);
                            
                            // 处理剧集数据
                            if (isset($movieData['vod_play_url']) && !empty($movieData['vod_play_url'])) {
                                $newEpisodesAdded = 0;
                                // 尝试处理分集数据，但捕获可能的异常以避免整个处理失败
                                try {
                                    processEpisodeData($movieData, $movieId, $pdo, $newEpisodesAdded);
                                } catch (Exception $e) {
                                    // 如果处理分集失败，记录错误但继续处理
                                    error_log('处理分集数据失败: ' . $e->getMessage());
                                }
                            }
                            
                            $pageSuccess++;
                            $totalSuccess++;
                        }
                    } catch (Exception $e) {
                        // 记录错误但继续处理下一个
                        error_log('处理电影数据失败: ' . $e->getMessage());
                        $pageFailed++;
                        $totalFailed++;
                        continue;
                    }
                }
                
                // 提交事务
                $pdo->commit();
                
                $pagesSuccess++;
                error_log("第{$pageNum}页采集完成！处理 {$pageProcessed} 个视频，成功 {$pageSuccess} 个，失败 {$pageFailed} 个，跳过 {$pageSkipped} 个重复项");
                
            } catch (Exception $e) {
                // 回滚事务
                if ($pdo->inTransaction()) {
                    $pdo->rollBack();
                }
                
                $pagesFailed++;
                error_log('第{$pageNum}页采集失败：' . $e->getMessage());
                
                // 如果是批量采集，继续处理下一页；如果是单页采集，则抛出异常
                if (!$isBatchCollect) {
                    throw $e;
                }
            }
        }
        
        if ($isBatchCollect) {
            $success = "批量采集完成！共采集 {$pagesSuccess} 页，失败 {$pagesFailed} 页，总共处理 {$totalProcessed} 个视频，成功 {$totalSuccess} 个，失败 {$totalFailed} 个，跳过 {$totalSkipped} 个重复项";
        } else {
            $success = "第{$pageStart}页采集完成！总共处理 {$totalProcessed} 个视频，成功 {$totalSuccess} 个，失败 {$totalFailed} 个，跳过 {$totalSkipped} 个重复项";
        }
        
    } catch (Exception $e) {
        // 回滚事务
        if ($pdo->inTransaction()) {
            $pdo->rollBack();
        }
        
        $error = '采集失败：' . $e->getMessage();
    }
}

/**
 * 检查是否存在相同的电影
 */
function checkExistingMovie($movieData, $pdo, $typeIdMapping, $categoryIdMapping) {
    // 映射数据
    $type = 1; // 默认类型
    if (isset($movieData['type_id_1']) && isset($typeIdMapping[$movieData['type_id_1']])) {
        $type = $typeIdMapping[$movieData['type_id_1']];
    }
    
    $title = isset($movieData['vod_name']) ? $movieData['vod_name'] : '';
    
    if (empty($title)) {
        throw new Exception('视频标题不能为空');
    }
    
    // 检查数据库中是否已存在相同标题和类型的电影
    $sql = "SELECT id FROM movies WHERE title = ? AND type = ? LIMIT 1";
    $stmt = $pdo->prepare($sql);
    $stmt->execute([$title, $type]);
    $result = $stmt->fetch(PDO::FETCH_ASSOC);
    
    return $result ? $result['id'] : null;
}

/**
 * 计算播放链接中的实际地址数量
 */
function calculateActualEpisodeCount($vodPlayUrl) {
    if (!isset($vodPlayUrl) || empty($vodPlayUrl)) {
        return 1; // 如果没有播放链接，默认为1集
    }
    
    // 提取$$$后面的内容
    $parts = explode('$$$', $vodPlayUrl);
    $urlContent = count($parts) > 1 ? $parts[1] : $vodPlayUrl;
    
    // 使用parseEpisodes函数来获取实际的剧集数量
    $episodes = parseEpisodes($urlContent);
    
    return !empty($episodes) ? count($episodes) : 1;
}

/**
 * 更新已存在的电影数据
 */
function updateExistingMovie($movieData, $movieId, $pdo, $typeIdMapping, $categoryIdMapping) {
    // 映射数据
    $type = 1; // 默认类型
    if (isset($movieData['type_id_1']) && isset($typeIdMapping[$movieData['type_id_1']])) {
        $type = $typeIdMapping[$movieData['type_id_1']];
    }
    
    $categoryId = 1; // 默认分类
    if (isset($movieData['type_id']) && isset($categoryIdMapping[$movieData['type_id']])) {
        $categoryId = $categoryIdMapping[$movieData['type_id']];
    }
    
    // 处理播放链接 - 修改为始终尝试提取第一个可播放URL作为主播放链接
    $playUrl = '';
    $totalEpisodes = isset($movieData['vod_total']) ? intval($movieData['vod_total']) : 1;
    
    // 如果vod_total为0，则计算实际的剧集数量
    if ($totalEpisodes === 0 && isset($movieData['vod_play_url'])) {
        $totalEpisodes = calculateActualEpisodeCount($movieData['vod_play_url']);
    }
    
    // 始终尝试提取第一个可播放URL作为主播放链接
    if (isset($movieData['vod_play_url']) && !empty($movieData['vod_play_url'])) {
        // 提取$$$后面的内容
        $parts = explode('$$$', $movieData['vod_play_url']);
        if (count($parts) > 1) {
            $urlContent = $parts[1];
            
            // 处理链接内容
            $playUrl = extractFirstPlayableUrl($urlContent);
        } else {
            // 如果没有$$$，则直接处理整个链接
            $playUrl = extractFirstPlayableUrl($movieData['vod_play_url']);
        }
    }
    
    // 准备更新数据
    $title = isset($movieData['vod_name']) ? $movieData['vod_name'] : '';
    $subtitle = '';
    if (isset($movieData['vod_sub'])) {
        $subtitle = $movieData['vod_sub'];
    } elseif (isset($movieData['vod_en'])) {
        $subtitle = $movieData['vod_en'];
    }
    
    $director = isset($movieData['vod_director']) ? $movieData['vod_director'] : '';
    $actors = isset($movieData['vod_actor']) ? $movieData['vod_actor'] : '';
    $region = isset($movieData['vod_area']) ? $movieData['vod_area'] : '';
    $language = isset($movieData['vod_lang']) ? $movieData['vod_lang'] : '';
    $releaseYear = isset($movieData['vod_year']) ? $movieData['vod_year'] : '';
    $coverImg = isset($movieData['vod_pic']) ? $movieData['vod_pic'] : '';
    $description = isset($movieData['vod_content']) ? $movieData['vod_content'] : '';
    
    // 更新电影数据
    $sql = "UPDATE movies SET title = ?, subtitle = ?, type = ?, category_id = ?, director = ?, actors = ?, region = ?, language = ?, release_year = ?, cover_img = ?, description = ?, play_url = ?, total_episodes = ? WHERE id = ?";
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute([$title, $subtitle, $type, $categoryId, $director, $actors, $region, $language, $releaseYear, $coverImg, $description, $playUrl, $totalEpisodes, $movieId]);
    
    return $movieId;
}

/**
 * 清空指定电影的所有剧集
 */
function clearExistingEpisodes($movieId, $pdo) {
    $sql = "DELETE FROM episodes WHERE movie_id = ?";
    $stmt = $pdo->prepare($sql);
    $stmt->execute([$movieId]);
}

/**
 * 处理电影数据并保存到数据库
 */
function processMovieData($movieData, $pdo, $typeIdMapping, $categoryIdMapping) {
    // 映射数据
    $type = 1; // 默认类型
    if (isset($movieData['type_id_1']) && isset($typeIdMapping[$movieData['type_id_1']])) {
        $type = $typeIdMapping[$movieData['type_id_1']];
    }
    
    $categoryId = 1; // 默认分类
    if (isset($movieData['type_id']) && isset($categoryIdMapping[$movieData['type_id']])) {
        $categoryId = $categoryIdMapping[$movieData['type_id']];
    }
    
    // 处理播放链接 - 修改为始终尝试提取第一个可播放URL作为主播放链接
    $playUrl = '';
    $totalEpisodes = isset($movieData['vod_total']) ? intval($movieData['vod_total']) : 1;
    
    // 如果vod_total为0，则计算实际的剧集数量
    if ($totalEpisodes === 0 && isset($movieData['vod_play_url'])) {
        $totalEpisodes = calculateActualEpisodeCount($movieData['vod_play_url']);
    }
    
    // 始终尝试提取第一个可播放URL作为主播放链接
    if (isset($movieData['vod_play_url']) && !empty($movieData['vod_play_url'])) {
        // 提取$$$后面的内容
        $parts = explode('$$$', $movieData['vod_play_url']);
        if (count($parts) > 1) {
            $urlContent = $parts[1];
            
            // 处理链接内容
            $playUrl = extractFirstPlayableUrl($urlContent);
        } else {
            // 如果没有$$$，则直接处理整个链接
            $playUrl = extractFirstPlayableUrl($movieData['vod_play_url']);
        }
    }
    
    // 准备插入数据
    $title = isset($movieData['vod_name']) ? $movieData['vod_name'] : '';
    $subtitle = '';
    if (isset($movieData['vod_sub'])) {
        $subtitle = $movieData['vod_sub'];
    } elseif (isset($movieData['vod_en'])) {
        $subtitle = $movieData['vod_en'];
    }
    
    $director = isset($movieData['vod_director']) ? $movieData['vod_director'] : '';
    $actors = isset($movieData['vod_actor']) ? $movieData['vod_actor'] : '';
    $region = isset($movieData['vod_area']) ? $movieData['vod_area'] : '';
    $language = isset($movieData['vod_lang']) ? $movieData['vod_lang'] : '';
    $releaseYear = isset($movieData['vod_year']) ? $movieData['vod_year'] : '';
    $coverImg = isset($movieData['vod_pic']) ? $movieData['vod_pic'] : '';
    $description = isset($movieData['vod_content']) ? $movieData['vod_content'] : '';
    
    // 验证必填字段
    if (empty($title)) {
        throw new Exception('视频标题不能为空');
    }
    
    // 插入电影数据
    $sql = "INSERT INTO movies (title, subtitle, type, category_id, director, actors, region, language, release_year, cover_img, description, play_url, total_episodes, status, create_time) 
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())";
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute([$title, $subtitle, $type, $categoryId, $director, $actors, $region, $language, $releaseYear, $coverImg, $description, $playUrl, $totalEpisodes, 1]);
    
    // 返回插入的电影ID
    return $pdo->lastInsertId();
}

/**
 * 处理剧集数据并保存到数据库
 */
function processEpisodeData($movieData, $movieId, $pdo, &$newEpisodesAdded) {
    $newEpisodesAdded = 0; // 初始化新增剧集计数
    
    if (!isset($movieData['vod_play_url']) || empty($movieData['vod_play_url'])) {
        throw new Exception('vod_play_url不存在或为空');
    }
    
    // 提取$$$后面的内容
    $parts = explode('$$$', $movieData['vod_play_url']);
    $urlContent = count($parts) > 1 ? $parts[1] : $movieData['vod_play_url'];
    
    // 解析剧集数据
    $episodes = parseEpisodes($urlContent);
    
    if (empty($episodes)) {
        throw new Exception('未找到有效的剧集数据');
    }
    
    // 插入分集数据
    foreach ($episodes as $episodeNum => $episodeData) {
        // 跳过空链接
        if (empty(trim($episodeData['url']))) {
            continue;
        }
        
        // 替换m3u8链接
        $playUrl = str_replace('index.m3u8', '3000k/hls/mixed.m3u8', $episodeData['url']);
        
        // 检查是否已存在相同的剧集
        if (checkExistingEpisode($movieId, $episodeNum, $pdo)) {
            continue; // 跳过已存在的剧集，但不计入跳过计数
        }
        
        // 准备SQL语句
        if (!empty($episodeData['title'])) {
            $sql = "INSERT INTO episodes (movie_id, episode_num, episode_title, play_url, create_time) VALUES (?, ?, ?, ?, NOW())";
            $stmt = $pdo->prepare($sql);
            $stmt->execute([$movieId, $episodeNum, $episodeData['title'], $playUrl]);
        } else {
            $sql = "INSERT INTO episodes (movie_id, episode_num, play_url, create_time) VALUES (?, ?, ?, NOW())";
            $stmt = $pdo->prepare($sql);
            $stmt->execute([$movieId, $episodeNum, $playUrl]);
        }
        
        $newEpisodesAdded++; // 增加新增剧集计数
    }
}

/**
 * 检查是否存在相同的剧集
 */
function checkExistingEpisode($movieId, $episodeNum, $pdo) {
    $sql = "SELECT id FROM episodes WHERE movie_id = ? AND episode_num = ? LIMIT 1";
    $stmt = $pdo->prepare($sql);
    $stmt->execute([$movieId, $episodeNum]);
    $result = $stmt->fetch(PDO::FETCH_ASSOC);
    
    return $result !== false;
}

/**
 * 解析剧集数据，支持#和$分隔符
 */
function parseEpisodes($urlContent) {
    $episodes = array();
    $episodeNum = 1;
    
    // 首先尝试用#分割（主要分隔符）
    $segments = explode('#', $urlContent);
    
    if (count($segments) > 1) {
        foreach ($segments as $segment) {
            if (empty(trim($segment))) {
                continue;
            }
            
            // 检查是否有$分隔的标题和链接
            if (strpos($segment, '$') !== false) {
                list($title, $url) = explode('$', $segment, 2);
                $episodes[$episodeNum++] = array(
                    'title' => trim($title),
                    'url' => trim($url)
                );
            } else {
                // 可能是单独的标题或链接
                $episodes[$episodeNum++] = array(
                    'title' => '',
                    'url' => trim($segment)
                );
            }
        }
    } else {
        // 如果#分割失败，尝试用$分割
        $segments = explode('$', $urlContent);
        
        for ($i = 0; $i < count($segments); $i += 2) {
            if (isset($segments[$i+1]) && !empty(trim($segments[$i+1]))) {
                $episodes[$episodeNum++] = array(
                    'title' => trim($segments[$i]),
                    'url' => trim($segments[$i+1])
                );
            }
        }
    }
    
    return $episodes;
}

/**
 * 提取第一个可播放的URL
 */
function extractFirstPlayableUrl($urlContent) {
    // 解析所有剧集
    $episodes = parseEpisodes($urlContent);
    
    // 返回第一个有效的URL
    foreach ($episodes as $episode) {
        if (!empty($episode['url'])) {
            // 替换m3u8链接
            return str_replace('index.m3u8', '3000k/hls/mixed.m3u8', $episode['url']);
        }
    }
    
    return '';
}
?>
<!DOCTYPE html>
<html lang="zh-CN">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>视频采集 - 爱影视</title>
    <link rel="stylesheet" href="css/style.css">
    <script src="js/main.js"></script>
</head>
<body>
    <?php include 'includes/header.php'; ?>
    
    <div class="container">
        <div class="content-wrapper">
            <?php include 'includes/sidebar.php'; ?>
            
            <div class="content">
                <div class="main-content">
                    <div class="breadcrumb">
                        <a href="index.php">首页</a> &gt; 
                        <span>视频采集</span>
                    </div>
                    
                    <div class="panel">
                        <div class="panel-header">
                            <h3>视频一键采集</h3>
                        </div>
                        
                        <div class="panel-body">
                            <?php if ($error): ?>
                                <div class="alert alert-error"><?php echo $error; ?></div>
                            <?php endif; ?>
                            
                            <?php if ($success): ?>
                                <div class="alert alert-success"><?php echo $success; ?></div>
                            <?php endif; ?>
                            
                            <!-- 一键采集表单 -->
                            <div class="form-container" style="text-align: center; padding: 30px;">
                                <p style="margin-bottom: 20px;">选择视频类型和分类后点击下方按钮开始采集视频数据</p>
                                <form method="POST" action="">
                                    <div style="margin-bottom: 20px;">
                                        <label for="video_type" style="display: inline-block; margin-right: 10px;">选择视频类型：</label>
                                        <select name="video_type" id="video_type" style="padding: 8px; border: 1px solid #ddd; border-radius: 4px;">
                                            <option value="0">全部类型</option>
                                            <?php foreach ($videoTypes as $type): ?>
                                                <option value="<?php echo $type['id']; ?>"><?php echo $type['name']; ?></option>
                                            <?php endforeach; ?>
                                        </select>
                                    </div>
                                    <div id="category_container" style="margin-bottom: 20px; display: none;">
                                        <label for="video_category" style="display: inline-block; margin-right: 10px;">选择视频分类：</label>
                                        <select name="video_category" id="video_category" style="padding: 8px; border: 1px solid #ddd; border-radius: 4px;">
                                            <option value="0">全部分类</option>
                                        </select>
                                    </div>
                                    <div style="margin-bottom: 20px;">
                                        <label for="page_start" style="display: inline-block; margin-right: 10px;">起始页码：</label>
                                        <input type="number" name="page_start" id="page_start" min="1" value="1" style="padding: 8px; border: 1px solid #ddd; border-radius: 4px; width: 80px;">
                                        <label for="page_end" style="display: inline-block; margin-left: 20px; margin-right: 10px;">结束页码：</label>
                                        <input type="number" name="page_end" id="page_end" min="1" value="1" style="padding: 8px; border: 1px solid #ddd; border-radius: 4px; width: 80px;">
                                        <span style="margin-left: 10px;">（每页约20条数据）</span>
                                    </div>
                                    <div style="margin-bottom: 20px;">
                                        <button type="submit" name="one_click_collect" class="btn-primary" style="padding: 10px 30px; font-size: 18px; margin-right: 15px;">采集单页</button>
                                        <button type="submit" name="batch_collect" class="btn-secondary" style="padding: 10px 30px; font-size: 18px;">批量采集（从起始页到结束页）</button>
                                    </div>
                                </form>
                                
                                <?php if ($totalProcessed > 0): ?>
                                    <div style="margin-top: 20px; text-align: left;">
                                        <h4>采集统计</h4>
                                        <ul>
                                            <li>总处理数: <?php echo $totalProcessed; ?></li>
                                            <li>成功数: <?php echo $totalSuccess; ?></li>
                                            <li>失败数: <?php echo $totalFailed; ?></li>
                                            <li>跳过重复: <?php echo $totalSkipped; ?></li>
                                        </ul>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <?php include 'includes/footer.php'; ?>
    
    <script>
    // 处理视频类型和分类的联动
    document.getElementById('video_type').addEventListener('change', function() {
        var selectedType = this.value;
        var categorySelect = document.getElementById('video_category');
        
        // 清空现有选项
        categorySelect.innerHTML = '<option value="0">全部分类</option>';
        
        // 如果选择了特定类型，添加该类型下的分类
        if (selectedType > 0 && typeof videoCategories !== 'undefined' && videoCategories[selectedType]) {
            videoCategories[selectedType].forEach(function(category) {
                var option = document.createElement('option');
                option.value = category.id;
                option.textContent = category.name;
                categorySelect.appendChild(option);
            });
        }
        
        // 显示或隐藏分类选择框
        if (selectedType > 0) {
            document.getElementById('category_container').style.display = 'block';
        } else {
            document.getElementById('category_container').style.display = 'none';
        }
    });
    
    // 初始化视频分类数据
    var videoCategories = <?php echo json_encode($categories); ?>;
    
    // 页面加载时触发一次类型选择事件，以初始化分类选择框
    document.addEventListener('DOMContentLoaded', function() {
        document.getElementById('video_type').dispatchEvent(new Event('change'));
    });
    </script>
</body>
</html>