<?php
// 视频类型管理页面
require_once 'includes/check_login.php';
require_once '../config.php';

// 初始化session，用于存储操作结果消息
session_start();

// 获取数据库连接
$pdo = getDbConnection();

// 检查session中的成功消息
if (isset($_SESSION['success_message']) && !empty($_SESSION['success_message'])) {
    $success = $_SESSION['success_message'];
    // 显示后清除session中的消息
    unset($_SESSION['success_message']);
}

// 处理单个删除操作
if (isset($_GET['action']) && $_GET['action'] == 'delete' && isset($_GET['id'])) {
    $id = intval($_GET['id']);
    
    // 检查是否有视频正在使用此类型
    $stmt = $pdo->prepare("SELECT COUNT(*) FROM movies WHERE type = ?");
    $stmt->execute([$id]);
    $count = $stmt->fetchColumn();
    
    if ($count > 0) {
        $error = '该类型下有视频数据，无法删除';
    } else {
        // 检查是否有分类正在使用此类型
        $stmt = $pdo->prepare("SELECT COUNT(*) FROM movie_categories WHERE type = ?");
        $stmt->execute([$id]);
        $count = $stmt->fetchColumn();
        
        if ($count > 0) {
            $error = '该类型下有分类数据，无法删除';
        } else {
            $stmt = $pdo->prepare("DELETE FROM video_types WHERE id = ?");
            $stmt->execute([$id]);
            $success = '视频类型删除成功';
        }
    }
}

// 处理批量删除操作
if (isset($_POST['action']) && $_POST['action'] == 'batch_delete') {
    if (isset($_POST['type_ids']) && !empty($_POST['type_ids'])) {
        $type_ids = $_POST['type_ids'];
        
        try {
            // 检查每个类型是否有相关数据
            $error_types = [];
            foreach ($type_ids as $id) {
                // 检查是否有视频正在使用此类型
                $stmt = $pdo->prepare("SELECT COUNT(*) FROM movies WHERE type = ?");
                $stmt->execute([$id]);
                $count = $stmt->fetchColumn();
                
                if ($count > 0) {
                    $stmt = $pdo->prepare("SELECT name FROM video_types WHERE id = ?");
                    $stmt->execute([$id]);
                    $type_name = $stmt->fetchColumn();
                    $error_types[] = $type_name . "(有视频数据)";
                    continue;
                }
                
                // 检查是否有分类正在使用此类型
                $stmt = $pdo->prepare("SELECT COUNT(*) FROM movie_categories WHERE type = ?");
                $stmt->execute([$id]);
                $count = $stmt->fetchColumn();
                
                if ($count > 0) {
                    $stmt = $pdo->prepare("SELECT name FROM video_types WHERE id = ?");
                    $stmt->execute([$id]);
                    $type_name = $stmt->fetchColumn();
                    $error_types[] = $type_name . "(有分类数据)";
                }
            }
            
            if (!empty($error_types)) {
                $error = "以下类型无法删除: " . implode(", ", $error_types);
            } else {
                // 批量删除类型
                $placeholders = implode(',', array_fill(0, count($type_ids), '?'));
                $sql = "DELETE FROM video_types WHERE id IN ($placeholders)";
                $stmt = $pdo->prepare($sql);
                $stmt->execute($type_ids);
                
                // 使用session存储成功消息
                $_SESSION['success_message'] = '批量删除成功';
                // 重定向到不带任何参数的页面
                header("Location: video_type_list.php");
                exit;
            }
        } catch (Exception $e) {
            $error = "批量删除失败：" . $e->getMessage();
        }
    }
}

// 处理批量更改状态操作
if (isset($_POST['action']) && $_POST['action'] == 'batch_change_status') {
    if (isset($_POST['type_ids']) && !empty($_POST['type_ids']) && isset($_POST['status'])) {
        $type_ids = $_POST['type_ids'];
        $status = intval($_POST['status']);
        
        try {
            $placeholders = implode(',', array_fill(0, count($type_ids), '?'));
            $sql = "UPDATE video_types SET status = ? WHERE id IN ($placeholders)";
            $stmt = $pdo->prepare($sql);
            // 构造参数数组，第一个是状态，后面是类型ID
            $params = array_merge([$status], $type_ids);
            $stmt->execute($params);
            
            // 使用session存储成功消息
            $_SESSION['success_message'] = '批量更改状态成功';
            // 重定向到不带任何参数的页面
            header("Location: video_type_list.php");
            exit;
        } catch (Exception $e) {
            $error = "批量更改状态失败：" . $e->getMessage();
        }
    }
}

// 获取视频类型列表（修改查询以包含分类数量）
$sql = "
    SELECT 
        vt.*, 
        COUNT(mc.id) AS category_count 
    FROM 
        video_types vt
    LEFT JOIN 
        movie_categories mc ON vt.id = mc.type
    GROUP BY 
        vt.id
    ORDER BY 
        vt.sort_order ASC
";
$videoTypes = $pdo->query($sql)->fetchAll();
?>
<!DOCTYPE html>
<html lang="zh-CN">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>视频类型管理 - 爱影视</title>
    <link rel="stylesheet" href="css/style.css">
    <script src="js/main.js"></script>
</head>
<body>
    <?php include 'includes/header.php'; ?>
    
    <div class="container">
        <div class="content-wrapper">
            <?php include 'includes/sidebar.php'; ?>
            
            <div class="content">
                <div class="main-content">
                    <div class="page-header">
                        <h2>视频类型管理</h2>
                        <p>注意！删除视频类型前先确认是否有分类数据，否则无法删除。</p>
                        <div class="header-actions">
                            <a href="video_type_edit.php" class="btn-primary">添加视频类型</a>
                        </div>
                    </div>
                    
                    <?php if (isset($error)): ?>
                        <div class="alert alert-error"><?php echo $error; ?></div>
                    <?php endif; ?>
                    
                    <?php if (isset($success)): ?>
                        <div class="alert alert-success"><?php echo $success; ?></div>
                    <?php endif; ?>
                    
                    <form id="batch-form" method="post" action="video_type_list.php">
                        <input type="hidden" name="action" id="batch-action" value="">
                        <input type="hidden" name="status" id="batch-status-value" value="">
                        
                        <table class="data-table">
                            <thead>
                                <tr>
                                    <th><input type="checkbox" id="select-all"></th>
                                    <th>ID</th>
                                    <th>类型名称</th>
                                    <th>类型代码</th>
                                    <th>排序</th>
                                    <th>状态</th>
                                    <th>分类数量</th>
                                    <th>操作</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if (empty($videoTypes)): ?>
                                    <tr>
                                        <td colspan="8" class="no-data">暂无数据</td>
                                    </tr>
                                <?php else: ?>
                                    <?php foreach ($videoTypes as $type): ?>
                                        <tr>
                                            <td><input type="checkbox" class="type-checkbox" name="type_ids[]" value="<?php echo $type['id']; ?>"></td>
                                            <td><?php echo $type['id']; ?></td>
                                            <td><?php echo $type['name']; ?></td>
                                            <td><?php echo $type['code']; ?></td>
                                            <td><?php echo $type['sort_order']; ?></td>
                                            <td><?php echo $type['status'] == 1 ? '<span class="status-active">启用</span>' : '<span class="status-inactive">禁用</span>'; ?></td>
                                            <td><?php echo $type['category_count']; ?></td>
                                            <td>
                                                <a href="video_type_edit.php?id=<?php echo $type['id']; ?>" class="btn-edit">编辑</a>
                                                <a href="video_type_list.php?action=delete&id=<?php echo $type['id']; ?>" class="btn-delete" onclick="return confirm('确定要删除该视频类型吗？');">删除</a>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </tbody>
                        </table>
                        
                        <!-- 批量操作容器 -->
                        <div class="batch-actions-container">
                            <div class="batch-actions">
                                <div class="batch-info">
                                    <input type="checkbox" id="select-all-bottom">
                                    <span>已选择 <span id="selected-count">0</span> 项</span>
                                </div>
                                <div class="batch-controls">
                                    <select id="batch-status-select">
                                        <option value="">选择状态</option>
                                        <option value="1">启用</option>
                                        <option value="0">禁用</option>
                                    </select>
                                    <button type="button" id="batch-status-btn" class="btn-edit">批量更改状态</button>
                                    <button type="button" id="batch-delete-btn" class="btn-delete">批量删除</button>
                                </div>
                            </div>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>
    
    <?php include 'includes/footer.php'; ?>
</body>
<script>
    // 全选/取消全选功能 - 顶部
    document.getElementById('select-all').addEventListener('change', function() {
        updateCheckboxes(this.checked);
        updateSelectedCount();
    });
    
    // 全选/取消全选功能 - 底部
    document.getElementById('select-all-bottom').addEventListener('change', function() {
        updateCheckboxes(this.checked);
        updateSelectedCount();
    });
    
    // 单个复选框更改事件
    document.querySelectorAll('.type-checkbox').forEach(checkbox => {
        checkbox.addEventListener('change', function() {
            updateSelectedCount();
            updateSelectAllCheckboxes();
        });
    });
    
    // 更新选中数量显示
    function updateSelectedCount() {
        const selectedCount = document.querySelectorAll('.type-checkbox:checked').length;
        document.getElementById('selected-count').textContent = selectedCount;
    }
    
    // 批量更改状态
    document.getElementById('batch-status-btn').addEventListener('click', function() {
        const selectedStatus = document.getElementById('batch-status-select').value;
        const selectedTypes = getSelectedTypes();
        
        if (selectedTypes.length === 0) {
            alert('请至少选择一个视频类型');
            return;
        }
        
        if (!selectedStatus) {
            alert('请选择要更改的状态');
            return;
        }
        
        document.getElementById('batch-action').value = 'batch_change_status';
        document.getElementById('batch-status-value').value = selectedStatus;
        document.getElementById('batch-form').submit();
    });
    
    // 批量删除
    document.getElementById('batch-delete-btn').addEventListener('click', function() {
        const selectedTypes = getSelectedTypes();
        
        if (selectedTypes.length === 0) {
            alert('请至少选择一个视频类型');
            return;
        }
        
        if (confirm('确定要删除选中的' + selectedTypes.length + '个视频类型吗？\n注意：有相关数据的类型将无法删除。')) {
            document.getElementById('batch-action').value = 'batch_delete';
            document.getElementById('batch-form').submit();
        }
    });
    
    // 获取选中的类型ID
    function getSelectedTypes() {
        const checkboxes = document.querySelectorAll('.type-checkbox:checked');
        const typeIds = Array.from(checkboxes).map(checkbox => checkbox.value);
        return typeIds;
    }
    
    // 初始化选中数量显示
    window.onload = function() {
        updateSelectedCount();
    };
    
    // 更新所有复选框状态
    function updateCheckboxes(checked) {
        const checkboxes = document.querySelectorAll('.type-checkbox');
        checkboxes.forEach(checkbox => {
            checkbox.checked = checked;
        });
    }
    
    // 更新全选复选框状态
    function updateSelectAllCheckboxes() {
        const allCheckboxes = document.querySelectorAll('.type-checkbox');
        const checkedCheckboxes = document.querySelectorAll('.type-checkbox:checked');
        const isAllChecked = allCheckboxes.length > 0 && allCheckboxes.length === checkedCheckboxes.length;
        const isNoneChecked = checkedCheckboxes.length === 0;
        
        // 更新顶部全选框
        const selectAllTop = document.getElementById('select-all');
        selectAllTop.checked = isAllChecked;
        selectAllTop.indeterminate = !isAllChecked && !isNoneChecked;
        
        // 更新底部全选框
        const selectAllBottom = document.getElementById('select-all-bottom');
        selectAllBottom.checked = isAllChecked;
        selectAllBottom.indeterminate = !isAllChecked && !isNoneChecked;
    }
</script>
</html>