<?php
// 引入数据库配置文件
require_once 'config.php';

// 获取数据库连接
$pdo = getDbConnection();

// 获取类型参数，确保它是整数（修复Array to string conversion错误）
$type = 1; // 默认值
if (isset($_GET['type'])) {
    // 确保$type始终是一个整数
    $type = is_array($_GET['type']) && isset($_GET['type'][0]) ? intval($_GET['type'][0]) : intval($_GET['type']);
} else {
    // 确保即使没有type参数，$type也是整数
    $type = 1;
}

// 额外的安全检查，确保$type始终是整数
$type = intval($type);
$typeName = '电影'; // 默认类型名称
$activePage = 'movies'; // 默认活动页面为电影

// 从数据库获取视频类型信息
$stmt = $pdo->prepare("SELECT name, code FROM video_types WHERE id = ? AND status = 1");
$stmt->execute([$type]);
$videoType = $stmt->fetch();

// 如果找不到对应的视频类型，只修改显示名称，不改变$type值
if ($videoType) {
    $typeName = $videoType['name'];
    $activePage = $videoType['code'];
} else {
    // 只更改显示名称，保留原始type值
    $typeName = '未找到的类型';
    $activePage = 'movies';
}

// 初始化分类名称变量
$categoryName = '';

// 设置页面标题和活动页面
$pageTitle = ($categoryName ? $categoryName . ' - ' : '') . $typeName . ' - 爱影视';

// 获取分类参数
$categoryId = isset($_GET['category_id']) ? intval($_GET['category_id']) : 0;

// 验证分类ID是否属于当前类型
if ($categoryId > 0) {
    $stmt = $pdo->prepare("SELECT COUNT(*) FROM movie_categories WHERE id = ? AND type = ?");
    $stmt->execute([$categoryId, $type]);
    if ($stmt->fetchColumn() == 0) {
        $categoryId = 0; // 如果分类不属于当前类型，重置为0
    }
}

// 获取所有分类
$stmt = $pdo->prepare("SELECT * FROM movie_categories WHERE type = ? AND status = 1");
$stmt->execute([$type]);
$categories = $stmt->fetchAll();

// 创建统一的状态过滤条件SQL片段
$statusFilterSql = "INNER JOIN video_types vt ON m.type = vt.id 
    INNER JOIN movie_categories mc ON m.category_id = mc.id 
    WHERE m.status = 1 
    AND vt.status = 1 
    AND mc.status = 1 
    AND m.type = ?";

// 基础SQL查询
$sql = "SELECT m.* FROM movies m $statusFilterSql";
$params = [$type];

// 如果选择了特定分类，添加分类过滤条件
if ($categoryId > 0) {
    $sql .= " AND mc.id = ?";
    $params[] = $categoryId;
}

// 默认按创建时间倒序排列
$sql .= " ORDER BY m.create_time DESC";

// 获取分页参数
$page = isset($_GET['page']) ? intval($_GET['page']) : 1;
$pageSize = 12;
$offset = ($page - 1) * $pageSize;

// 获取总记录数 - 使用统一的状态过滤逻辑
$countSql = "SELECT COUNT(m.id) AS count FROM movies m $statusFilterSql";

// 如果选择了特定分类，添加分类过滤条件
if ($categoryId > 0) {
    $countSql .= " AND mc.id = ?";
}

$stmt = $pdo->prepare($countSql);
$stmt->execute($params);
$totalCount = $stmt->fetchColumn();
$totalPages = ceil($totalCount / $pageSize);

// 添加分页限制
$sql .= " LIMIT $offset, $pageSize";

// 获取电影列表
$stmt = $pdo->prepare($sql);
$stmt->execute($params);
$movies = $stmt->fetchAll();

// 获取当前分类名称
if ($categoryId > 0) {
    $stmt = $pdo->prepare("SELECT name FROM movie_categories WHERE id = ?");
    $stmt->execute([$categoryId]);
    $category = $stmt->fetch();
    $categoryName = $category ? $category['name'] : '';
    
    // 更新页面标题
    $pageTitle = $categoryName . ' - ' . $typeName . ' - 电影网站';
}

// 引入共用的头部 - 修复路径错误
require_once 'includes/header.php';
?>
<div class="container mx-auto px-4 py-8 max-w-7xl">
    <!-- 页面标题 -->
    <h2 class="text-3xl font-bold mb-6"><?php echo $categoryName ? $categoryName . ' - ' : ''; ?><?php echo $typeName; ?></h2>
    
    <!-- 分类筛选 -->
    <div class="category-filter bg-dark-light p-4 rounded-lg mb-6">
        <ul class="flex flex-wrap gap-2">
            <li><a href="category.php?type=<?php echo intval($type); ?>" class="px-3 text-sm py-1.5 rounded-full text-primary hover:bg-primary hover:text-white transition-all <?php echo $categoryId == 0 ? 'bg-primary text-white' : 'bg-secondary'; ?>">全部分类</a></li>
            <?php foreach ($categories as $category): ?>
                <li><a href="category.php?type=<?php echo intval($type); ?>&category_id=<?php echo $category['id']; ?>" class="px-3 text-sm py-1.5 rounded-full text-primary hover:bg-primary hover:text-white transition-all <?php echo $categoryId == $category['id'] ? 'bg-primary text-white' : 'bg-secondary'; ?>">
                    <?php echo $category['name']; ?>
                </a></li>
            <?php endforeach; ?>
        </ul>
    </div>

    <!-- 电影列表 -->
    <div class="movie-list grid grid-cols-2 sm:grid-cols-3 md:grid-cols-4 lg:grid-cols-6 gap-4 mb-8">
        <?php if (empty($movies)): ?>
            <p class="no-data col-span-full text-center py-10 text-muted">暂无数据</p>
        <?php else: ?>
            <?php foreach ($movies as $movie): ?>
                <div class="movie-item relative h-64 rounded-lg overflow-hidden shadow hover:-translate-y-2 hover:shadow-lg transition-all duration-300 bg-dark-light">
                    <a href="play.php?id=<?php echo $movie['id']; ?>" class="block h-full">
                        <img src="<?php echo $movie['cover_img'] ?: 'images/default_cover.jpg'; ?>" alt="<?php echo $movie['title']; ?>" class="absolute inset-0 w-full h-full object-cover transition-transform duration-500 hover:scale-110">
                        <div class="absolute top-2 right-2 bg-black/70 text-white text-xs px-2 py-1 rounded-full">
                            热度: <?php echo $movie['view_count']; ?>
                        </div>
                        <div class="absolute bottom-0 left-0 right-0 bg-gradient-to-t from-black/90 to-transparent p-3 text-white">
                            <h4 class="text-base font-medium line-clamp-1 mb-1"><?php echo $movie['title']; ?></h4>
                            <p class="text-xs text-gray-300 line-clamp-1 mb-1 opacity-80"><?php echo $movie['director']; ?> | <?php echo $movie['actors']; ?></p>
                            <div class="flex justify-between items-center">
                                <p class="text-xs text-gray-400 opacity-80 m-0"><?php echo $movie['release_year']; ?></p>
                                <p class="text-xs text-gray-400 opacity-80 m-0">共<?php echo $movie['total_episodes']; ?>集</p>
                            </div>
                        </div>
                    </a>
                </div>
            <?php endforeach; ?>
        <?php endif; ?>
    </div>

    <!-- 分页 -->
    <?php if ($totalPages > 1): ?>
        <div class="pagination flex justify-center mt-8 space-x-2">
            <a href="category.php?type=<?php echo intval($type); ?><?php echo $categoryId ? '&category_id=' . $categoryId : ''; ?>&page=1" class="px-4 py-2 bg-secondary rounded-md text-primary hover:bg-primary hover:text-white transition-all <?php echo $page == 1 ? 'opacity-50 cursor-not-allowed' : ''; ?>" <?php echo $page == 1 ? 'aria-disabled="true"' : ''; ?>>首页</a>
            <a href="category.php?type=<?php echo intval($type); ?><?php echo $categoryId ? '&category_id=' . $categoryId : ''; ?>&page=<?php echo $page - 1; ?>" class="px-4 py-2 bg-secondary rounded-md text-primary hover:bg-primary hover:text-white transition-all <?php echo $page == 1 ? 'opacity-50 cursor-not-allowed' : ''; ?>" <?php echo $page == 1 ? 'aria-disabled="true"' : ''; ?>>上一页</a>
            
            <?php for ($i = max(1, $page - 2); $i <= min($totalPages, $page + 2); $i++): ?>
                <a href="category.php?type=<?php echo intval($type); ?><?php echo $categoryId ? '&category_id=' . $categoryId : ''; ?>&page=<?php echo $i; ?>" class="px-4 py-2 rounded-md text-white transition-all <?php echo $page == $i ? 'bg-primary' : 'bg-secondary text-primary hover:bg-primary'; ?>"><?php echo $i; ?></a>
            <?php endfor; ?>
            
            <a href="category.php?type=<?php echo intval($type); ?><?php echo $categoryId ? '&category_id=' . $categoryId : ''; ?>&page=<?php echo $page + 1; ?>" class="px-4 py-2 bg-secondary rounded-md text-primary hover:bg-primary hover:text-white transition-all <?php echo $page == $totalPages ? 'opacity-50 cursor-not-allowed' : ''; ?>" <?php echo $page == $totalPages ? 'aria-disabled="true"' : ''; ?>>下一页</a>
            <a href="category.php?type=<?php echo intval($type); ?><?php echo $categoryId ? '&category_id=' . $categoryId : ''; ?>&page=<?php echo $totalPages; ?>" class="px-4 py-2 bg-secondary rounded-md text-primary hover:bg-primary hover:text-white transition-all <?php echo $page == $totalPages ? 'opacity-50 cursor-not-allowed' : ''; ?>" <?php echo $page == $totalPages ? 'aria-disabled="true"' : ''; ?>>末页</a>
        </div>
    <?php endif; ?>
</div>
<?php
// 引入页脚
require_once 'includes/footer.php';
?>