<?php
// 检查是否已安装程序
$isInstalled = false;

// 1. 检查配置文件是否存在
if (file_exists('config.php')) {
    // 尝试包含配置文件
    try {
        require_once 'config.php';
        
        // 2. 尝试连接数据库
        $pdo = getDbConnection();
        
        // 3. 检查是否存在必要的表
        $stmt = $pdo->query("SHOW TABLES LIKE 'admin_users'");
        if ($stmt->rowCount() > 0) {
            $isInstalled = true;
        }
    } catch (Exception $e) {
        // 配置文件存在但无法连接数据库或缺少必要的表
        $isInstalled = false;
    }
}

// 如果未安装，跳转到安装页面
if (!$isInstalled && basename($_SERVER['PHP_SELF']) !== 'install.php') {
    header('Location: install.php');
    exit;
}

// 引入数据库配置文件（已存在且验证通过后）
if ($isInstalled) {
    require_once 'config.php';
}

// 设置页面标题和活动页面
$pageTitle = '爱影视 - 首页';
$activePage = 'index';

// 获取排序方式参数，默认为按最新更新
$sortBy = isset($_GET['sort']) && $_GET['sort'] === 'hot' ? 'hot' : 'latest';

// 获取数据库连接
$pdo = getDbConnection();

// 创建统一的状态过滤条件SQL片段
$statusFilterSql = "INNER JOIN video_types vt ON m.type = vt.id 
    INNER JOIN movie_categories mc ON m.category_id = mc.id 
    WHERE m.status = 1 
    AND vt.status = 1 
    AND mc.status = 1";

// 获取轮播图数据 - 使用统一的状态过滤条件
$stmt = $pdo->prepare("SELECT b.* FROM banner b 
    INNER JOIN movies m ON b.movie_id = m.id 
    $statusFilterSql 
    ORDER BY b.sort_order ASC");
$stmt->execute();
$banners = $stmt->fetchAll();

// 如果没有轮播图数据，默认使用最近更新的视频 - 使用统一的状态过滤条件
if (empty($banners)) {
    $latestMovies = $pdo->query("SELECT m.* FROM movies m 
        $statusFilterSql 
        ORDER BY m.create_time DESC LIMIT 5")->fetchAll();
    $banners = array_map(function($movie) {
        return [
            'id' => $movie['id'],
            'movie_id' => $movie['id'],
            'title' => $movie['title'],
            'description' => $movie['description'],
            'image_url' => $movie['cover_img'] ?: 'images/default_cover.jpg',
        ];
    }, $latestMovies);
}

// 从数据库动态获取所有启用的视频类型
$videoTypes = [];
$typeCategories = [];
$typeMovies = [];

try {
    // 获取所有启用状态的视频类型
    $stmt = $pdo->prepare("SELECT * FROM video_types WHERE status = 1 ORDER BY sort_order ASC");
    $stmt->execute();
    $videoTypes = $stmt->fetchAll();
    
    // 为每种视频类型获取分类和视频数据
    foreach ($videoTypes as $type) {
        $typeId = $type['id'];
        
        // 获取该类型下的所有分类
        $stmt = $pdo->prepare("SELECT * FROM movie_categories WHERE type = :typeId AND status = 1");
        $stmt->bindParam(':typeId', $typeId, PDO::PARAM_INT);
        $stmt->execute();
        $typeCategories[$typeId] = $stmt->fetchAll();
        
        // 根据排序方式决定ORDER BY子句
        $orderByClause = $sortBy === 'hot' ? 'ORDER BY m.view_count DESC' : 'ORDER BY m.create_time DESC';
        
        // 获取该类型下的视频 - 根据选择的排序方式排序
        $stmt = $pdo->prepare("SELECT m.* FROM movies m 
            $statusFilterSql 
            AND m.type = :typeId 
            $orderByClause LIMIT 12");
        $stmt->bindParam(':typeId', $typeId, PDO::PARAM_INT);
        $stmt->execute();
        $typeMovies[$typeId] = $stmt->fetchAll();
    }
} catch (PDOException $e) {
    // 出错时使用默认视频类型
    $videoTypes = [
        ['id' => 1, 'name' => '电影'],
        ['id' => 2, 'name' => '剧集'],
        ['id' => 3, 'name' => '动漫'],
        ['id' => 4, 'name' => '综艺'],
        ['id' => 5, 'name' => '纪录片'],
        ['id' => 6, 'name' => '其他类型']
    ];
    
    // 初始化默认数据
    foreach ($videoTypes as $type) {
        $typeCategories[$type['id']] = [];
        $typeMovies[$type['id']] = [];
    }
}

// 引入共用的头部
require_once 'includes/header.php';
?>
    <!-- 轮播图 -->
    <section class="banner mb-8 relative overflow-hidden">
        <?php if (!empty($banners)): ?>
            <div class="banner-slider">
                <?php foreach ($banners as $index => $banner): ?>
                    <div class="banner-slide <?php echo $index === 0 ? 'active' : ''; ?> absolute inset-0 transition-opacity duration-1000 ease-in-out">
                        <a href="play.php?id=<?php echo $banner['movie_id']; ?>" class="block h-full">
                            <img src="<?php echo $banner['image_url']; ?>" alt="<?php echo $banner['title']; ?>" class="absolute inset-0 w-full h-full object-cover">
                            <div class="absolute inset-0 bg-black/60"></div>
                            <div class="banner-content text-center z-10 relative h-full flex flex-col items-center justify-center text-light px-4">
                                <h2 class="text-4xl font-bold mb-4 transform transition-transform hover:scale-105 duration-500"><?php echo $banner['title']; ?></h2>
                                <?php if (!empty($banner['description'])): ?>
                                    <p class="text-xl line-clamp-2 max-w-3xl mb-6 overflow-hidden text-overflow-ellipsis whitespace-normal h-16"><?php echo $banner['description']; ?></p>
                                <?php endif; ?>
                            </div>
                        </a>
                    </div>
                <?php endforeach; ?>
                
                <!-- 轮播图指示器 -->
                <div class="banner-indicators absolute bottom-4 left-0 right-0 flex justify-center gap-2 z-20">
                    <?php foreach ($banners as $index => $banner): ?>
                        <button class="banner-indicator w-3 h-3 rounded-full bg-white/50 hover:bg-white transition-all duration-300 <?php echo $index === 0 ? 'active bg-white scale-125' : ''; ?>" data-index="<?php echo $index; ?>"></button>
                    <?php endforeach; ?>
                </div>
            </div>
        <?php else: ?>
            <!-- 默认轮播图 -->
            <div class="bg-banner bg-cover bg-center flex items-center justify-center text-light relative">
                <div class="absolute inset-0"></div>
                <div class="banner-content text-center z-10">
                    <h2 class="text-4xl font-bold mb-4">欢迎来到爱影视</h2>
                    <p class="text-xl">海量影视资源，免费观看</p>
                </div>
            </div>
        <?php endif; ?>
    </section>
    
    <!-- 排序切换按钮组 -->
    <div class="sort-controls flex justify-center mb-8 w-full px-4">
        <div class="inline-flex rounded-md shadow-sm w-full max-w-xs justify-center" role="group">
            <button id="sort-latest" class="sort-btn px-6 py-2.5 text-sm font-medium rounded-l-lg border border-primary transition-colors duration-300 focus:outline-none
                <?php echo $sortBy === 'latest' ? 'bg-primary text-white hover:bg-primary/90' : 'bg-dark-light text-primary hover:bg-primary hover:text-white'; ?>"
                data-sort="latest">
                最近更新
            </button>
            <button id="sort-hot" class="sort-btn px-6 py-2.5 text-sm font-medium rounded-r-lg border border-primary transition-colors duration-300 focus:outline-none
                <?php echo $sortBy === 'hot' ? 'bg-primary text-white hover:bg-primary/90' : 'bg-dark-light text-primary hover:bg-primary hover:text-white'; ?>"
                data-sort="hot">
                热度排行
            </button>
        </div>
    </div>

    <!-- 动态生成视频类型区块 - 添加ID以便通过JS更新 -->
    <div id="movie-content">
    <!-- 动态生成视频类型区块 -->
    <?php $firstSection = true; ?>
    <?php foreach ($videoTypes as $type): ?>
        <?php $typeId = $type['id']; ?>
        <?php $typeMoviesList = $typeMovies[$typeId] ?? []; ?>
        
        <?php if (!empty($typeMoviesList)): ?>
            <section class="movie-section<?php echo $firstSection ? '' : ' mt-10'; ?>">
                <?php $firstSection = false; ?>
                <div class="flex flex-wrap items-center justify-between gap-2 mb-4">
                    <h3 class="text-2xl font-bold">
                        <?php echo $sortBy === 'hot' ? '热门' : '最新'; ?><?php echo htmlspecialchars($type['name']); ?>
                    </h3>
                    <div class="flex-1">
                        <ul class="flex flex-wrap gap-2 justify-end">
                            <?php if (isset($typeCategories[$typeId]) && !empty($typeCategories[$typeId])): ?>
                                <?php foreach ($typeCategories[$typeId] as $category): ?>
                                    <li><a href="category.php?type=<?php echo $category['type']; ?>&category_id=<?php echo $category['id']; ?>" class="px-3 py-1 text-sm bg-secondary rounded-full text-primary hover:bg-primary hover:text-white transition-all">
                                        <?php echo htmlspecialchars($category['name']); ?>
                                    </a></li>
                                <?php endforeach; ?>
                            <?php endif; ?>
                            <li><a href="category.php?type=<?php echo $typeId; ?>" class="px-3 py-1 text-sm bg-secondary rounded-full text-primary hover:bg-primary hover:text-white transition-all">全部</a></li>
                        </ul>
                    </div>
                </div>
                <div class="movie-list grid grid-cols-2 sm:grid-cols-3 md:grid-cols-4 lg:grid-cols-6 gap-4">
                    <?php foreach ($typeMoviesList as $movie): ?>
                        <div class="movie-item relative h-64 rounded-lg overflow-hidden shadow hover:-translate-y-2 hover:shadow-lg transition-all duration-300 bg-dark-light">
                            <a href="play.php?id=<?php echo $movie['id']; ?>" class="block h-full">
                                <img src="<?php echo $movie['cover_img'] ?: 'images/default_cover.jpg'; ?>" alt="<?php echo $movie['title']; ?>" class="absolute inset-0 w-full h-full object-cover transition-transform duration-500 hover:scale-110">
                                <div class="absolute top-2 right-2 bg-black/70 text-white text-xs px-2 py-1 rounded-full">
                                    热度: <?php echo $movie['view_count']; ?>
                                </div>
                                <div class="absolute bottom-0 left-0 right-0 bg-gradient-to-t from-black/90 to-transparent p-3 text-white">
                                    <h4 class="text-base font-medium line-clamp-1 mb-1"><?php echo $movie['title']; ?></h4>
                                    <p class="text-xs text-gray-300 line-clamp-1 mb-1 opacity-80"><?php echo $movie['director']; ?> | <?php echo $movie['actors']; ?></p>
                                    <div class="flex justify-between items-center">
                                        <p class="text-xs text-gray-400 opacity-80 m-0"><?php echo $movie['release_year']; ?></p>
                                        <p class="text-xs text-gray-400 opacity-80 m-0">共<?php echo $movie['total_episodes']; ?>集</p>
                                    </div>
                                </div>
                            </a>
                        </div>
                    <?php endforeach; ?>
                </div>
            </section>
        <?php endif; ?>
    <?php endforeach; ?>
    </div>

<?php
// 引入共用的页脚
require_once 'includes/footer.php';
?>