// 播放页专用评论区AJAX处理文件

// 全局变量存储当前页码和电影ID
let currentPage = 1;
let currentMovieId = null;
const commentsPerPage = 10;

// 页面加载完成后执行
document.addEventListener('DOMContentLoaded', function() {
    // 添加回复相关样式
    addReplyStyles();
    
    // AJAX评论提交功能
    const commentForm = document.querySelector('.comment-form form');
    if (commentForm) {
        commentForm.addEventListener('submit', function(e) {
            e.preventDefault();
            
            // 获取表单数据
            const username = document.querySelector('input[name="username"]').value;
            const content = document.querySelector('textarea[name="content"]').value;
            const movieId = getMovieIdFromUrl();
            // 获取回复的评论ID（如果有）
            const replyToId = commentForm.getAttribute('data-reply-to');
            const replyToUsername = commentForm.getAttribute('data-reply-username');
            
            // 表单验证
            if (!username || !content || !movieId) {
                showMessage('昵称和评论内容不能为空', 'error');
                return;
            }
            
            // 创建AJAX请求
            const xhr = new XMLHttpRequest();
            xhr.open('POST', 'comment_submit_ajax.php', true);
            xhr.setRequestHeader('Content-Type', 'application/x-www-form-urlencoded');
            
            xhr.onload = function() {
                if (xhr.status === 200) {
                    try {
                        const response = JSON.parse(xhr.responseText);
                        
                        if (response.success) {
                            showMessage(response.message, 'success');
                            // 清空表单
                            commentForm.reset();
                            // 清除回复状态
                            commentForm.removeAttribute('data-reply-to');
                            commentForm.removeAttribute('data-reply-username');
                            // 重置页码并重新加载评论列表
                            currentPage = 1;
                            loadComments(movieId, true);
                        } else {
                            showMessage(response.message, 'error');
                        }
                    } catch (e) {
                        showMessage('服务器响应格式错误', 'error');
                    }
                } else {
                    showMessage('网络请求失败，请重试', 'error');
                }
            };
            
            xhr.onerror = function() {
                showMessage('网络连接错误，请重试', 'error');
            };
            
            // 发送请求，包含回复的评论ID（如果有）
            let requestBody = `movie_id=${movieId}&username=${encodeURIComponent(username)}&content=${encodeURIComponent(content)}`;
            if (replyToId) {
                requestBody += `&parent_id=${replyToId}&reply_to_username=${encodeURIComponent(replyToUsername)}`;
            }
            xhr.send(requestBody);
        });
    }
    
    // 初始加载评论（如果在播放页面）
    if (document.querySelector('.comments-section')) {
        const movieId = getMovieIdFromUrl();
        if (movieId) {
            currentMovieId = movieId;
            loadComments(movieId, true);
        }
    }
});

// 获取URL中的电影ID
function getMovieIdFromUrl() {
    const urlParams = new URLSearchParams(window.location.search);
    return urlParams.get('id');
}

// 加载评论列表 - 添加参数控制是否重置列表
function loadComments(movieId, resetList = false) {
    // 如果是重置列表，设置为第一页
    if (resetList) {
        currentPage = 1;
        currentMovieId = movieId;
    }
    
    const xhr = new XMLHttpRequest();
    // 添加分页参数
    xhr.open('GET', `comments_load_ajax.php?movie_id=${movieId}&page=${currentPage}&limit=${commentsPerPage}`, true);
    
    xhr.onload = function() {
        if (xhr.status === 200) {
            try {
                const response = JSON.parse(xhr.responseText);
                
                if (response.success) {
                    // 按父评论ID对评论进行分组，构建嵌套结构
                    const commentsMap = {};
                    const rootComments = [];
                    
                    // 先将所有评论存入map
                    response.comments.forEach(comment => {
                        comment.replies = [];
                        commentsMap[comment.id] = comment;
                    });
                    
                    // 构建嵌套结构
                    response.comments.forEach(comment => {
                        if (comment.parent_id && comment.parent_id > 0) {
                            // 这是一个回复
                            if (commentsMap[comment.parent_id]) {
                                commentsMap[comment.parent_id].replies.push(comment);
                            } else {
                                // 如果父评论不存在（可能被删除），将其作为根评论显示
                                rootComments.push(comment);
                            }
                        } else {
                            // 这是一个根评论
                            rootComments.push(comment);
                        }
                    });
                    
                    // 更新评论列表，传入是否重置和是否有更多评论的参数
                    updateCommentsList(rootComments, response.comment_count, resetList, response.has_more);
                } else {
                    showMessage(response.message, 'error');
                }
            } catch (e) {
                showMessage('评论加载失败', 'error');
            }
        }
    };
    
    xhr.onerror = function() {
        showMessage('网络连接错误，评论加载失败', 'error');
    };
    
    xhr.send();
}

// 更新评论列表显示 - 添加参数支持增量加载和显示更多按钮
function updateCommentsList(comments, commentCount, resetList = false, hasMore = false) {
    const commentsList = document.querySelector('.comments-list');
    const commentsTitle = document.querySelector('.comments-section h3');
    const commentForm = document.querySelector('.comment-form form');
    const usernameInput = document.querySelector('input[name="username"]');
    const contentTextarea = document.querySelector('textarea[name="content"]');

    if (commentsList) {
        // 更新评论数量
        if (commentsTitle) {
            commentsTitle.textContent = `评论 (${commentCount})`;
        }

        // 如果是重置列表，则清空现有评论
        if (resetList) {
            commentsList.innerHTML = '';
        }

        // 如果没有评论
        if (comments.length === 0 && resetList) {
            commentsList.innerHTML = `
                <div class="no-comments flex flex-col items-center justify-center p-10 bg-dark-light rounded-lg text-center">
                    <p class="text-light text-xl mb-2"><i class="fas fa-comment text-muted"></i> 暂无评论</p>
                    <p class="text-muted text-sm">快来发表第一条评论吧！</p>
                </div>
            `;
            return;
        }

        // 添加评论
        comments.forEach(comment => {
            const commentItem = createCommentElement(comment);
            commentsList.appendChild(commentItem);
        });
        
        // 移除旧的"显示更多"按钮
        const oldLoadMoreBtn = document.querySelector('.load-more-comments');
        if (oldLoadMoreBtn) {
            oldLoadMoreBtn.remove();
        }
        
        // 如果还有更多评论，添加"显示更多"按钮
        if (hasMore) {
            const loadMoreBtn = document.createElement('div');
            loadMoreBtn.className = 'load-more-comments flex justify-center mt-6 mb-4';
            loadMoreBtn.innerHTML = `
                <button class="btn btn-primary px-6 py-2 rounded-md hover:bg-primary/80 transition-colors">
                    显示更多评论
                </button>
            `;
            
            commentsList.appendChild(loadMoreBtn);
            
            // 添加点击事件
            loadMoreBtn.querySelector('button').addEventListener('click', function() {
                currentPage++;
                loadComments(currentMovieId, false);
            });
        }
    }
}

// 创建评论元素（支持嵌套回复）
function createCommentElement(comment, level = 0) {
    const commentItem = document.createElement('div');
    commentItem.className = 'comment-item bg-dark-light p-5 rounded-lg shadow-sm mb-4 border border-gray-700 hover:shadow-md transition-shadow duration-200';

    // 获取IP归属地
    const ipLocation = getIpLocation(comment.ip_address);

    // 格式化日期时间
    const formattedDate = formatCommentDate(comment.create_time);

    // 生成用户名首字母作为头像
    const avatarText = comment.username.charAt(0).toUpperCase();
    const avatarColor = getAvatarColor(comment.username);

    // 检查是否是回复评论
    let displayContent = htmlEscape(comment.content);
    if (comment.parent_id && comment.parent_id > 0 && comment.reply_to_username) {
        // 检查内容是否已经包含@用户名，如果不包含才添加
        const atUsername = `@${comment.reply_to_username}`;
        if (!displayContent.trim().toLowerCase().startsWith(atUsername.toLowerCase())) {
            displayContent = `<span class="reply-tag">${atUsername}</span> ${displayContent}`;
        }
    }

    // 第一层评论总是显示回复按钮
    commentItem.innerHTML = `
        <div class="comment-header flex items-start gap-3 mb-3">
            <div class="comment-avatar flex-shrink-0 w-10 h-10 rounded-full ${avatarColor} flex items-center justify-center text-white font-medium">
                ${avatarText}
            </div>
            <div class="comment-meta flex-grow">
                <div class="flex flex-wrap items-center gap-2 mb-1">
                    <span class="comment-username font-medium text-light">${htmlEscape(comment.username)}</span>
                    <span class="comment-ip text-xs text-muted">${ipLocation}</span>
                </div>
                <div class="comment-time text-xs text-muted">${formattedDate}</div>
            </div>
            <div class="comment-actions flex-shrink-0">
                <button class="reply-btn text-xs text-primary hover:text-primary/80 transition-colors" data-id="${comment.id}" data-username="${htmlEscape(comment.username)}">
                    回复
                </button>
            </div>
        </div>
        <div class="comment-content text-light leading-relaxed pl-13">
            ${displayContent}
        </div>
        <div class="replies pl-13 mt-4 space-y-3">
            <!-- 回复将在这里动态添加 -->
        </div>
    `;

    // 添加回复按钮事件
    const replyBtn = commentItem.querySelector('.reply-btn');
    const commentForm = document.querySelector('.comment-form form');
    const contentTextarea = document.querySelector('textarea[name="content"]');
    
    if (replyBtn && commentForm && contentTextarea) {
        replyBtn.addEventListener('click', function() {
            const commentId = this.getAttribute('data-id');
            const replyToUsername = this.getAttribute('data-username');
            
            // 存储当前回复的评论ID和用户名
            commentForm.setAttribute('data-reply-to', commentId);
            commentForm.setAttribute('data-reply-username', replyToUsername);
            
            // 聚焦到评论框并显示提示
            contentTextarea.focus();
            showMessage(`正在回复 @${replyToUsername} 的评论`, 'info');
            
            // 如果内容框为空，可以预先填入@用户名
            if (!contentTextarea.value.trim()) {
                contentTextarea.value = `@${replyToUsername} `;
            }
            
            // 滚动到评论表单
            commentForm.scrollIntoView({ behavior: 'smooth' });
        });
    }

    // 添加回复列表，传递当前层级+1
    const repliesContainer = commentItem.querySelector('.replies');
    if (repliesContainer && comment.replies && comment.replies.length > 0) {
        comment.replies.forEach(reply => {
            const replyElement = createReplyElement(reply, level + 1);
            repliesContainer.appendChild(replyElement);
        });
    }

    return commentItem;
}

// 创建回复元素
function createReplyElement(reply, level = 1) {
    const replyItem = document.createElement('div');
    replyItem.className = 'reply-item p-3 bg-secondary rounded-md border-l-2 border-primary';

    // 格式化日期时间
    const formattedDate = formatCommentDate(reply.create_time);

    // 生成用户名首字母作为头像
    const avatarText = reply.username.charAt(0).toUpperCase();
    const avatarColor = getAvatarColor(reply.username);

    // 检查是否是回复评论 - 修复双重@问题
    let displayContent = htmlEscape(reply.content);
    if (reply.parent_id && reply.parent_id > 0 && reply.reply_to_username) {
        // 检查内容是否已经包含@用户名，如果不包含才添加
        const atUsername = `@${reply.reply_to_username}`;
        if (!displayContent.trim().toLowerCase().startsWith(atUsername.toLowerCase())) {
            displayContent = `<span class="reply-tag">${atUsername}</span> ${displayContent}`;
        }
    }

    // 检查是否超过5层嵌套，如果超过则不显示回复按钮
    const shouldShowReplyBtn = level < 5;
    const replyButtonHTML = shouldShowReplyBtn ? 
        `<button class="reply-btn text-xs text-primary hover:text-primary/80 transition-colors" data-id="${reply.id}" data-username="${htmlEscape(reply.username)}">
            回复
        </button>` : '';

    replyItem.innerHTML = `
        <div class="reply-header flex items-start gap-2 mb-2">
            <div class="reply-avatar flex-shrink-0 w-6 h-6 rounded-full ${avatarColor} flex items-center justify-center text-white text-xs font-medium">
                ${avatarText}
            </div>
            <div class="reply-meta flex-grow">
                <div class="flex flex-wrap items-center gap-2 mb-1">
                    <span class="reply-username font-medium text-light text-sm">${htmlEscape(reply.username)}</span>
                    <span class="reply-time text-xs text-muted">${formattedDate}</span>
                </div>
            </div>
            <div class="reply-actions flex-shrink-0">
                ${replyButtonHTML}
            </div>
        </div>
        <div class="reply-content text-light leading-relaxed pl-8 text-sm">
            ${displayContent}
        </div>
        <div class="replies pl-8 mt-3 space-y-3">
            <!-- 嵌套回复将在这里动态添加 -->
        </div>
    `;

    // 如果没有超过嵌套层数限制，添加回复按钮事件
    if (shouldShowReplyBtn) {
        const replyBtn = replyItem.querySelector('.reply-btn');
        const commentForm = document.querySelector('.comment-form form');
        const contentTextarea = document.querySelector('textarea[name="content"]');
        
        if (replyBtn && commentForm && contentTextarea) {
            replyBtn.addEventListener('click', function() {
                const commentId = this.getAttribute('data-id');
                const replyToUsername = this.getAttribute('data-username');
                
                // 存储当前回复的评论ID和用户名
                commentForm.setAttribute('data-reply-to', commentId);
                commentForm.setAttribute('data-reply-username', replyToUsername);
                
                // 聚焦到评论框并显示提示
                contentTextarea.focus();
                showMessage(`正在回复 @${replyToUsername} 的评论`, 'info');
                
                // 如果内容框为空，可以预先填入@用户名
                if (!contentTextarea.value.trim()) {
                    contentTextarea.value = `@${replyToUsername} `;
                }
                
                // 滚动到评论表单
                commentForm.scrollIntoView({ behavior: 'smooth' });
            });
        }
    }

    // 添加嵌套回复列表，传递当前层级+1
    const repliesContainer = replyItem.querySelector('.replies');
    if (repliesContainer && reply.replies && reply.replies.length > 0) {
        reply.replies.forEach(nestedReply => {
            const nestedReplyElement = createReplyElement(nestedReply, level + 1);
            repliesContainer.appendChild(nestedReplyElement);
        });
    }

    return replyItem;
}

// 添加一些基础CSS样式来美化回复盖楼效果
function addReplyStyles() {
    if (document.getElementById('reply-styles')) return;
    
    const style = document.createElement('style');
    style.id = 'reply-styles';
    style.textContent = `
        .reply-tag {
            color: #007bff;
            font-weight: 500;
        }
        .comment-item {
            position: relative;
        }
        .replies {
            margin-left: 2rem;
            border-left: 2px solid rgba(0, 123, 255, 0.2);
            padding-left: 1rem;
        }
        .reply-item {
            margin-top: 0.75rem;
            background-color: rgba(33, 37, 41, 0.5);
            position: relative;
        }
        .reply-item::before {
            content: '';
            position: absolute;
            top: 1rem;
            left: -1.75rem;
            width: 1rem;
            height: 2px;
            background-color: rgba(0, 123, 255, 0.2);
        }
        .comment-form {
            margin-bottom: 2rem;
        }
    `;
    
    document.head.appendChild(style);
}

// 其他辅助函数保持不变
// ... 原有的formatCommentDate、getAvatarColor、getIpLocation、showMessage、showLoading、htmlEscape函数

// 格式化评论日期
function formatCommentDate(dateString) {
    const date = new Date(dateString);
    const now = new Date();
    const diffTime = Math.abs(now - date);
    const diffDays = Math.floor(diffTime / (1000 * 60 * 60 * 24));
    
    if (diffDays === 0) {
        // 今天的评论，显示时间
        return date.toLocaleTimeString('zh-CN', { hour: '2-digit', minute: '2-digit' });
    } else if (diffDays === 1) {
        // 昨天的评论
        return `昨天 ${date.toLocaleTimeString('zh-CN', { hour: '2-digit', minute: '2-digit' })}`;
    } else if (diffDays < 7) {
        // 一周内的评论
        return `${diffDays}天前`;
    } else {
        // 超过一周的评论，显示完整日期
        return date.toLocaleDateString('zh-CN');
    }
}

// 根据用户名生成头像颜色
function getAvatarColor(username) {
    const colors = [
        'bg-blue-500', 'bg-green-500', 'bg-purple-500', 'bg-orange-500',
        'bg-red-500', 'bg-indigo-500', 'bg-pink-500', 'bg-teal-500',
        'bg-amber-500', 'bg-cyan-500'
    ];
    
    // 根据用户名生成一个相对稳定的索引
    let hash = 0;
    for (let i = 0; i < username.length; i++) {
        hash = username.charCodeAt(i) + ((hash << 5) - hash);
    }
    
    const index = Math.abs(hash) % colors.length;
    return colors[index];
}

// 获取IP归属地信息
function getIpLocation(ipAddress) {
    if (!ipAddress) {
        return '未知IP';
    }
    
    // 检查是否为IPv4地址（包含点号）
    const ipParts = ipAddress.split('.');
    if (ipParts.length === 4) {
        // 对于IPv4地址，显示前两段，后两段用星号替代
        return `${ipParts[0]}.${ipParts[1]}.**.**`;
    }
    // 检查是否为IPv6地址（包含冒号）
    else if (ipAddress.includes(':')) {
        // 对于IPv6地址，显示前4段，后4段用星号替代
        const ipv6Parts = ipAddress.split(':');
        if (ipv6Parts.length >= 4) {
            // 保留前4段，其余用星号替代
            const maskedParts = ipv6Parts.slice(0, 4).concat(Array(4).fill('****'));
            return maskedParts.join(':');
        }
    }
    
    // 对于其他格式，返回原始地址
    return ipAddress;
}

// 显示消息提示
function showMessage(message, type = 'info') {
    // 移除已有的消息
    const existingMessage = document.querySelector('.ajax-message');
    if (existingMessage) {
        existingMessage.remove();
    }
    
    // 创建消息元素
    const messageElement = document.createElement('div');
    messageElement.className = `ajax-message alert alert-${type}`;
    messageElement.style.cssText = `
        position: fixed;
        top: 20px;
        right: 20px;
        padding: 15px 20px;
        border-radius: 5px;
        z-index: 1000;
        box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        max-width: 300px;
    `;
    
    // 根据类型设置样式
    if (type === 'success') {
        messageElement.style.backgroundColor = '#d4edda';
        messageElement.style.color = '#155724';
    } else if (type === 'error') {
        messageElement.style.backgroundColor = '#f8d7da';
        messageElement.style.color = '#721c24';
    } else {
        messageElement.style.backgroundColor = '#d1ecf1';
        messageElement.style.color = '#0c5460';
    }
    
    messageElement.textContent = message;
    
    // 添加到页面
    document.body.appendChild(messageElement);
    
    // 3秒后自动消失
    setTimeout(() => {
        messageElement.style.transition = 'opacity 0.5s, transform 0.5s';
        messageElement.style.opacity = '0';
        messageElement.style.transform = 'translateX(100%)';
        
        setTimeout(() => {
            messageElement.remove();
        }, 500);
    }, 3000);
}

// 显示加载状态
function showLoading(show) {
    let loadingElement = document.querySelector('.loading-overlay');
    
    if (show) {
        if (!loadingElement) {
            loadingElement = document.createElement('div');
            loadingElement.className = 'loading-overlay';
            loadingElement.style.cssText = `
                position: fixed;
                top: 0;
                left: 0;
                width: 100%;
                height: 100%;
                background-color: rgba(0,0,0,0.7);
                display: flex;
                align-items: center;
                justify-content: center;
                z-index: 9999;
            `;
            
            const spinner = document.createElement('div');
            spinner.style.cssText = `
                width: 40px;
                height: 40px;
                border: 4px solid #f3f3f3;
                border-top: 4px solid #337ab7;
                border-radius: 50%;
                animation: spin 1s linear infinite;
            `;
            
            const style = document.createElement('style');
            style.textContent = `
                @keyframes spin {
                    0% { transform: rotate(0deg); }
                    100% { transform: rotate(360deg); }
                }
            `;
            
            document.head.appendChild(style);
            loadingElement.appendChild(spinner);
            document.body.appendChild(loadingElement);
        } else {
            loadingElement.style.display = 'flex';
        }
    } else if (loadingElement) {
        loadingElement.style.display = 'none';
    }
}

// HTML转义函数
function htmlEscape(str) {
    const div = document.createElement('div');
    div.textContent = str;
    return div.innerHTML;
}