// 网站主JavaScript文件
console.log('爱影视 v1.0.11-25100092241');
// 页面加载完成后执行
window.addEventListener('DOMContentLoaded', function() {
    // 平滑滚动
    document.querySelectorAll('a[href^="#"]').forEach(anchor => {
        anchor.addEventListener('click', function (e) {
            e.preventDefault();
            
            document.querySelector(this.getAttribute('href'))?.scrollIntoView({
                behavior: 'smooth'
            });
        });
    });
    
    // 如果是播放页面，初始化剧集加载功能
    if (window.location.pathname.includes('play.php')) {
        setupEpisodeListeners();
    }
    
    // 初始化排序按钮
    setupSortButtons();
    
    // 轮播图脚本
    const slides = document.querySelectorAll('.banner-slide');
    const indicators = document.querySelectorAll('.banner-indicator');
    const bannerSlider = document.querySelector('.banner-slider');
    let currentIndex = 0;
    let slideInterval;
    let touchStartX = 0;
    let touchEndX = 0;
    const SWIPE_THRESHOLD = 50; // 滑动阈值

    // 初始化轮播图
    if (slides.length > 1) {
        // 设置自动轮播
        startSlideInterval();

        // 点击指示器切换轮播图
        indicators.forEach(indicator => {
            indicator.addEventListener('click', function() {
                const index = parseInt(this.getAttribute('data-index'));
                goToSlide(index);
                resetSlideInterval();
            });
        });

        // 添加移动端滑动支持
        if (bannerSlider) {
            bannerSlider.addEventListener('touchstart', handleTouchStart, false);
            bannerSlider.addEventListener('touchmove', handleTouchMove, false);
            bannerSlider.addEventListener('touchend', handleTouchEnd, false);
        }
    }

    // 切换到指定轮播图
    function goToSlide(index) {
        slides.forEach(slide => slide.classList.remove('active'));
        indicators.forEach(indicator => indicator.classList.remove('active', 'bg-white', 'scale-125'));

        slides[index].classList.add('active');
        indicators[index].classList.add('active', 'bg-white', 'scale-125');
        currentIndex = index;
    }

    // 下一张轮播图
    function nextSlide() {
        let nextIndex = currentIndex + 1;
        if (nextIndex >= slides.length) {
            nextIndex = 0;
        }
        goToSlide(nextIndex);
    }

    // 上一张轮播图
    function prevSlide() {
        let prevIndex = currentIndex - 1;
        if (prevIndex < 0) {
            prevIndex = slides.length - 1;
        }
        goToSlide(prevIndex);
    }

    // 开始自动轮播
    function startSlideInterval() {
        slideInterval = setInterval(nextSlide, 5000); // 5秒切换一次
    }

    // 重置自动轮播
    function resetSlideInterval() {
        clearInterval(slideInterval);
        startSlideInterval();
    }

    // 触摸开始事件
    function handleTouchStart(e) {
        touchStartX = e.changedTouches[0].screenX;
        // 触摸时暂停自动轮播
        clearInterval(slideInterval);
    }

    // 触摸移动事件
    function handleTouchMove(e) {
        touchEndX = e.changedTouches[0].screenX;
    }

    // 触摸结束事件
    function handleTouchEnd() {
        const diff = touchStartX - touchEndX;
        // 判断滑动方向和距离是否达到阈值
        if (diff > SWIPE_THRESHOLD) {
            // 向左滑动，显示下一张
            nextSlide();
        } else if (diff < -SWIPE_THRESHOLD) {
            // 向右滑动，显示上一张
            prevSlide();
        }
        // 无论是否切换轮播图，都重置自动轮播
        resetSlideInterval();
    }
});

// 排序按钮功能
function setupSortButtons() {
    const sortButtons = document.querySelectorAll('.sort-btn');
    const movieContent = document.getElementById('movie-content');
    
    if (sortButtons.length > 0 && movieContent) {
        sortButtons.forEach(button => {
            button.addEventListener('click', function() {
                const sortType = this.getAttribute('data-sort');
                loadSortContent(sortType);
            });
        });
    }
}

// 加载排序内容的统一函数
function loadSortContent(sortType) {
    // 移除注释，启用加载状态显示
    showLoading(true);
    
    // 发送AJAX请求
    const xhr = new XMLHttpRequest();
    xhr.open('GET', `ajax_sort.php?sort=${sortType}`, true);
    xhr.setRequestHeader('X-Requested-With', 'XMLHttpRequest');
    
    xhr.onload = function() {
        if (xhr.status >= 200 && xhr.status < 400) {
            try {
                const response = JSON.parse(xhr.responseText);
                
                if (response.success) {
                    // 更新按钮样式
                    const sortButtons = document.querySelectorAll('.sort-btn');
                    sortButtons.forEach(btn => {
                        const btnSortType = btn.getAttribute('data-sort');
                        if (btnSortType === sortType) {
                            btn.classList.remove('bg-dark-light', 'text-primary', 'hover:bg-primary', 'hover:text-white');
                            btn.classList.add('bg-primary', 'text-white', 'hover:bg-primary/90');
                        } else {
                            btn.classList.remove('bg-primary', 'text-white', 'hover:bg-primary/90');
                            btn.classList.add('bg-dark-light', 'text-primary', 'hover:bg-primary', 'hover:text-white');
                        }
                    });
                    
                    // 添加淡出效果
                    const movieContent = document.getElementById('movie-content');
                    movieContent.style.opacity = '0';
                    movieContent.style.transition = 'opacity 0.3s ease';
                    
                    // 等待淡出效果完成后更新内容
                    setTimeout(() => {
                        movieContent.innerHTML = response.content;
                        // 添加淡入效果
                        movieContent.style.opacity = '1';
                        
                        // 内容加载完成后隐藏加载状态
                        showLoading(false);
                    }, 300);
                    
                } else {
                    showMessage('加载失败: ' + (response.error || '未知错误'), 'error');
                    // 请求失败时隐藏加载状态
                    showLoading(false);
                }
            } catch (e) {
                showMessage('数据解析错误', 'error');
                console.error('JSON解析错误:', e);
                // 解析错误时隐藏加载状态
                showLoading(false);
            }
        } else {
            showMessage('服务器错误', 'error');
            // 服务器错误时隐藏加载状态
            showLoading(false);
        }
    };
    
    xhr.onerror = function() {
        showMessage('网络错误', 'error');
        // 网络错误时隐藏加载状态
        showLoading(false);
    };
    
    xhr.send();
};

// 显示加载状态
function showLoading(show) {
    let loadingElement = document.querySelector('.loading-overlay');
    
    if (show) {
        if (!loadingElement) {
            loadingElement = document.createElement('div');
            loadingElement.className = 'loading-overlay';
            loadingElement.style.cssText = `
                position: fixed;
                top: 0;
                left: 0;
                width: 100%;
                height: 100%;
                background-color: rgba(0,0,0,0.7);
                display: flex;
                align-items: center;
                justify-content: center;
                z-index: 9999;
            `;
            
            const spinner = document.createElement('div');
            spinner.style.cssText = `
                width: 40px;
                height: 40px;
                border: 4px solid #f3f3f3;
                border-top: 4px solid #337ab7;
                border-radius: 50%;
                animation: spin 1s linear infinite;
            `;
            
            const style = document.createElement('style');
            style.textContent = `
                @keyframes spin {
                    0% { transform: rotate(0deg); }
                    100% { transform: rotate(360deg); }
                }
            `;
            
            document.head.appendChild(style);
            loadingElement.appendChild(spinner);
            document.body.appendChild(loadingElement);
        } else {
            loadingElement.style.display = 'flex';
        }
    } else if (loadingElement) {
        loadingElement.style.display = 'none';
    }
}

// 显示消息提示
function showMessage(message, type = 'info') {
    // 移除已有的消息
    const existingMessage = document.querySelector('.ajax-message');
    if (existingMessage) {
        existingMessage.remove();
    }
    
    // 创建消息元素
    const messageElement = document.createElement('div');
    messageElement.className = `ajax-message alert alert-${type}`;
    messageElement.style.cssText = `
        position: fixed;
        top: 20px;
        right: 20px;
        padding: 15px 20px;
        border-radius: 5px;
        z-index: 1000;
        box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        max-width: 300px;
    `;
    
    // 根据类型设置样式
    if (type === 'success') {
        messageElement.style.backgroundColor = '#d4edda';
        messageElement.style.color = '#155724';
    } else if (type === 'error') {
        messageElement.style.backgroundColor = '#f8d7da';
        messageElement.style.color = '#721c24';
    } else {
        messageElement.style.backgroundColor = '#d1ecf1';
        messageElement.style.color = '#0c5460';
    }
    
    messageElement.textContent = message;
    
    // 添加到页面
    document.body.appendChild(messageElement);
    
    // 3秒后自动消失
    setTimeout(() => {
        messageElement.style.transition = 'opacity 0.5s, transform 0.5s';
        messageElement.style.opacity = '0';
        messageElement.style.transform = 'translateX(100%)';
        
        setTimeout(() => {
            messageElement.remove();
        }, 500);
    }, 3000);
}

// 处理浏览器前进后退事件
window.addEventListener('popstate', function(event) {
    if (event.state && event.state.sort) {
        // 直接调用统一的加载排序内容函数
        loadSortContent(event.state.sort);
    }
});

// ==================================================================
// 以下是新增的剧集加载相关功能
// ==================================================================

/**
 * 设置分集点击事件监听
 */
function setupEpisodeListeners() {
    const episodeItems = document.querySelectorAll('.episode-item');
    episodeItems.forEach(item => {
        item.addEventListener('click', (e) => {
            e.preventDefault();
            const episodeId = item.getAttribute('data-episode-id');
            const movieId = item.getAttribute('data-movie-id');
            loadEpisode(movieId, episodeId, item);
        });
    });
    
    // 页面加载完成后，确保当前选中的分集按钮在容器可视范围内
    setTimeout(() => {
        const activeEpisode = document.querySelector('.episode-item.bg-primary.text-white');
        if (activeEpisode) {
            ensureElementInViewport(activeEpisode);
        }
    }, 100);
}

/**
 * 加载分集
 */
function loadEpisode(movieId, episodeId, clickedItem) {
    // 发送AJAX请求获取剧集信息
    const xhr = new XMLHttpRequest();
    xhr.open('GET', `episode_load_ajax.php?movie_id=${movieId}&episode_id=${episodeId}`, true);
    
    xhr.onload = () => {
        if (xhr.status === 200) {
            try {
                const response = JSON.parse(xhr.responseText);
                
                if (response.success) {
                    // 更新选中样式
                    document.querySelectorAll('.episode-item').forEach(item => {
                        item.classList.remove('bg-primary', 'text-white');
                        item.classList.add('bg-secondary');
                    });
                    clickedItem.classList.remove('bg-secondary');
                    clickedItem.classList.add('bg-primary', 'text-white');
                    
                    // 确保当前选中的分集按钮在容器可视范围内
                    ensureElementInViewport(clickedItem);
                    
                    // 构建新的页面标题
                    // 优先使用响应中的movie_title，这样更准确
                    const movieTitle = response.movie_title || document.title.split(' - ')[0];
                    let episodeInfo = `第${response.episode_num}集`;
                    if (response.episode_title && response.episode_title.trim() !== '') {
                        episodeInfo = response.episode_title;
                    }
                    const newPageTitle = `${movieTitle} ${episodeInfo} - 爱影视`;
                    
                    // 更新浏览器标题
                    document.title = newPageTitle;
                    
                    // 切换到新剧集
                    if (typeof videoPlayer !== 'undefined' && videoPlayer.switchEpisode) {
                        videoPlayer.switchEpisode(response.play_url, response.episode_num, response.episode_title, response.movie_title);
                        // 保留main.js中的提示显示
                        setTimeout(() => {
                            showMessage(`正在播放${episodeInfo}`, 'info');
                        }, 500);
                    } else if (typeof videoPlayer !== 'undefined' && videoPlayer.playVideo) {
                        // 回退到playVideo方法
                        videoPlayer.playVideo(response.play_url);
                        // 延迟显示消息
                        setTimeout(() => {
                            showMessage(`正在播放${episodeInfo}`, 'info');
                        }, 500);
                    } else {
                        // 最底层回退
                        const videoElement = document.querySelector('video');
                        if (videoElement) {
                            videoElement.pause();
                            videoElement.src = response.play_url + '?t=' + new Date().getTime();
                            videoElement.load();
                            videoElement.play();
                        }
                        showMessage(`正在播放${episodeInfo}`, 'info');
                    }
                    
                    // 更新URL但不刷新页面
                    const newUrl = `play.php?id=${movieId}&episode_id=${episodeId}`;
                    window.history.pushState({ path: newUrl }, '', newUrl);
                } else {
                    showMessage(response.message || '加载剧集失败', 'error');
                }
            } catch (e) {
                console.error('解析响应错误:', e);
                showMessage('服务器响应格式错误', 'error');
            }
        } else {
            showMessage('网络请求失败，请重试', 'error');
        }
    };
    
    xhr.onerror = () => {
        showMessage('网络连接错误，请重试', 'error');
    };
    
    xhr.send();
}

/**
 * 确保元素在其父容器的可视范围内，但不改变当前页面的滚动位置
 * 只在元素不在可视范围内时进行最小幅度的滚动
 */
function ensureElementInViewport(element) {
    // 获取元素的父容器（分集列表容器）
    const container = element.closest('.episodes-grid');
    if (!container) return;
    
    const elementRect = element.getBoundingClientRect();
    const containerRect = container.getBoundingClientRect();
    
    // 检查元素是否完全在容器可视范围内
    const isInView = (
        elementRect.top >= containerRect.top &&
        elementRect.bottom <= containerRect.bottom &&
        elementRect.left >= containerRect.left &&
        elementRect.right <= containerRect.right
    );
    
    // 如果元素不在可视范围内，则进行滚动
    if (!isInView) {
        // 计算需要滚动的位置
        let scrollLeft = container.scrollLeft;
        let scrollTop = container.scrollTop;
        
        // 水平方向调整
        if (elementRect.left < containerRect.left) {
            // 元素在容器左侧外面，向左滚动
            scrollLeft -= (containerRect.left - elementRect.left) + 10; // 加10px的边距
        } else if (elementRect.right > containerRect.right) {
            // 元素在容器右侧外面，向右滚动
            scrollLeft += (elementRect.right - containerRect.right) + 10; // 加10px的边距
        }
        
        // 垂直方向调整（如果需要）
        if (elementRect.top < containerRect.top) {
            // 元素在容器顶部外面，向上滚动
            scrollTop -= (containerRect.top - elementRect.top) + 10; // 加10px的边距
        } else if (elementRect.bottom > containerRect.bottom) {
            // 元素在容器底部外面，向下滚动
            scrollTop += (elementRect.bottom - containerRect.bottom) + 10; // 加10px的边距
        }
        
        // 使用平滑滚动到计算的位置
        container.scrollTo({
            left: scrollLeft,
            top: scrollTop,
            behavior: 'smooth'
        });
    }
}