/**
 * 视频播放器模块 - 优化重构版
 * 功能：视频播放、控制、快捷键、历史记录、全屏模式、画中画和下载等
 */
class VideoPlayer {
    constructor() {
        // 配置常量
        this.config = {
            playbackRates: [0.5, 0.75, 1.0, 1.25, 1.5, 2.0, 3.0, 5.0, 10.0],
            defaultRateIndex: 2, // 默认 1.0x
            historySaveInterval: 5, // 历史记录保存间隔(秒)
            minSaveTime: 5, // 最小保存时间(秒)
            fullscreenZIndex: 9999,
            controlZIndex: 10000,
            preloadSize: 1 * 1, // 预加载大小 (1B)
            preloadDelay: 500 // 视频开始播放后延迟预加载的时间(毫秒)
        };

        // DOM元素缓存
        this.elements = {
            video: null,
            playerContainer: null,
            historyPrompt: null,
            historyTimeSpan: null,
            resumeBtn: null,
            closeHistoryBtn: null,
            rateIndicator: null,
            currentRate: null,
            exitFullscreenHint: null,
            shortcutHelp: null,
            fullscreenContainer: null,
            downloadModal: null,
            downloadFrameContainer: null
        };

        // 播放相关状态
        this.state = {
            videoUrl: '',
            hls: null,
            currentPlayingUrl: '',
            currentVideoType: null, // 'm3u8' 或 'binary'
            currentRateIndex: this.config.defaultRateIndex,
            isFullscreenMode: false,
            scrollTop: 0,
            originalVideoStyle: null,
            originalParent: null,
            originalNextSibling: null,
            originalControlStyles: null,
            preloadingUrl: null,
            lastHistorySaveTime: 0, // 添加上次保存时间记录
            userStartedPlaying: false // 用户是否已开始播放
        };

        // 存储相关
        this.contentTypeCache = {};
        // 新增：用于存储已预加载的分集信息
        this.preloadedEpisodes = new Set();
        // 预加载任务队列
        this.preloadQueue = [];
        // 当前是否有预加载任务在执行
        this.isPreloading = false;
        // 预加载超时ID
        this.preloadTimeoutId = null;
        // 当前正在播放的剧集ID
        this.currentEpisodeId = null;
        
        // 初始化
        this.init();
    }

    /**
     * 初始化播放器
     */
    init() {
        // 绑定DOMContentLoaded事件，确保DOM加载完毕后再执行
        document.addEventListener('DOMContentLoaded', () => {
            this.setupDOMReferences();
            if (this.elements.video) {
                this.createPlayerUI();
                this.setupEventListeners();
                this.loadVideoSource();
            }
        });
    }

    /**
     * 设置DOM元素引用
     */
    setupDOMReferences() {
        this.elements.video = document.querySelector('video');
        this.elements.downloadModal = document.getElementById('downloadModal');
        this.elements.downloadFrameContainer = document.getElementById('downloadFrameContainer');
    }

    /**
     * 创建播放器UI元素
     */
    createPlayerUI() {
        // 找到视频所在的容器
        const videoSection = this.elements.video.parentElement;
        this.elements.playerContainer = videoSection;
        
        // 为视频容器添加相对定位
        videoSection.style.position = 'relative';
        
        // 创建观看历史弹窗
        this.elements.historyPrompt = document.createElement('div');
        this.elements.historyPrompt.className = 'history-prompt fixed top-1/2 left-1/2 transform -translate-x-1/2 -translate-y-1/2 bg-dark-light p-6 rounded-lg shadow-lg z-50 hidden';
        this.elements.historyPrompt.innerHTML = `
            <div class="prompt-text text-white text-lg mb-4">上次看到 <span id="historyTime">00:00</span>，是否继续？</div>
            <div class="flex justify-end gap-3">
                <button id="resumeBtn" class="prompt-btn px-4 py-2 bg-primary text-white rounded-md hover:bg-primary/90 transition-colors">继续</button>
                <button id="closeHistoryBtn" class="close-btn px-4 py-2 bg-secondary text-white rounded-md hover:bg-secondary/90 transition-colors">重新播放</button>
            </div>
        `;
        document.body.appendChild(this.elements.historyPrompt);
        
        // 获取历史相关元素引用
        this.elements.historyTimeSpan = document.getElementById('historyTime');
        this.elements.resumeBtn = document.getElementById('resumeBtn');
        this.elements.closeHistoryBtn = document.getElementById('closeHistoryBtn');
        
        // 创建播放速率指示器
        this.elements.rateIndicator = document.createElement('div');
        this.elements.rateIndicator.className = 'rate-indicator absolute top-4 right-4 bg-black/75 text-white px-3 py-1 rounded-md z-40 hidden';
        this.elements.rateIndicator.innerHTML = `
            <i class="fas fa-running mr-1"></i>
            <span>播放速率: <span id="currentRate">1.0</span>x</span>
        `;
        videoSection.appendChild(this.elements.rateIndicator);
        this.elements.currentRate = document.getElementById('currentRate');
        
        // 创建退出全屏提示
        this.elements.exitFullscreenHint = document.createElement('div');
        this.elements.exitFullscreenHint.className = 'exit-fullscreen-hint fixed top-4 left-1/2 transform -translate-x-1/2 bg-black/75 text-white px-4 py-2 rounded-md z-50 hidden';
        this.elements.exitFullscreenHint.textContent = '按 F 键切换视频窗口大小，按 ESC 退出全屏';
        document.body.appendChild(this.elements.exitFullscreenHint);
        
        // 创建全屏容器
        this.elements.fullscreenContainer = document.createElement('div');
        this.elements.fullscreenContainer.id = 'fullscreen-container';
        this.elements.fullscreenContainer.style.position = 'fixed';
        this.elements.fullscreenContainer.style.top = '0';
        this.elements.fullscreenContainer.style.left = '0';
        this.elements.fullscreenContainer.style.width = '100vw';
        this.elements.fullscreenContainer.style.height = '100vh';
        this.elements.fullscreenContainer.style.zIndex = this.config.fullscreenZIndex;
        this.elements.fullscreenContainer.style.display = 'none';
        this.elements.fullscreenContainer.style.backgroundColor = 'black';
        document.body.appendChild(this.elements.fullscreenContainer);
    }

    /**
     * 加载视频源
     */
    loadVideoSource() {
        // 获取视频源
        const sourceElement = this.elements.video.querySelector('source');
        if (!sourceElement) return;

        this.state.videoUrl = sourceElement.src;
        sourceElement.remove(); // 移除source标签，手动控制视频源

        // 播放视频
        this.playVideo(this.state.videoUrl);
    }

    /**
     * 设置事件监听器
     */
    setupEventListeners() {
        // 键盘快捷键 - 整合所有键盘事件处理
        document.addEventListener('keydown', (e) => {
            // 仅当焦点不在输入元素上时才处理快捷键
            if (e.target.tagName === 'INPUT' || e.target.tagName === 'TEXTAREA') {
                return;
            }

            // 如果焦点在视频元素上，且按下的是空格键，则不处理（避免与浏览器默认行为冲突）
            if (e.target === this.elements.video && e.key === ' ') {
                return;
            }

            // 阻止默认行为（特别是空格键会导致页面滚动）
            e.preventDefault();

            // 优先处理空格键
            if (e.key === ' ' && this.elements.playerContainer) {
                e.stopPropagation();
                
                if (this.elements.historyPrompt && !this.elements.historyPrompt.classList.contains('hidden')) {
                    if (this.elements.resumeBtn) this.elements.resumeBtn.click();
                } else {
                    this.togglePlayPause();
                }
                return;
            }

            // 观看历史提示打开时：Esc 放弃历史，重新从头播放
            if (this.elements.historyPrompt && !this.elements.historyPrompt.classList.contains('hidden') && e.key === 'Escape') {
                e.stopPropagation();
                // 设置当前时间为0表示重新播放
                if (this.elements.video) {
                    this.elements.video.currentTime = 0;
                    // 明确触发播放，确保ESC键能正确执行重新播放功能
                    this.elements.video.play().catch(err => {
                        console.log('自动播放被浏览器阻止:', err);
                    });
                }
                // 隐藏历史提示
                this.hideHistoryPrompt();
                return;
            }

            // 下载弹窗打开时：只处理 ESC 关闭弹窗
            if (this.elements.downloadModal && this.elements.downloadModal.style.display === 'flex' && e.key === 'Escape') {
                e.stopPropagation();
                this.closeDownloadModal();
                return;
            }

            // 其他快捷键处理
            this.handleKeyboardShortcuts(e);
        }, true);

        // 视频播放进度监控 - 优化历史记录保存
        this.elements.video.addEventListener('timeupdate', this.debounce(() => {
            // 每N秒保存一次观看历史
            if (this.state.currentPlayingUrl) {
                // 即使在防抖期间，也确保在整数秒边界保存
                if (Math.floor(this.elements.video.currentTime) % this.config.historySaveInterval === 0) {
                    this.saveHistory(this.state.currentPlayingUrl, this.elements.video.currentTime);
                }
            }
        }, 500));

        // 添加标志，表示用户是否已开始播放
        this.elements.video.addEventListener('play', () => {
            if (this.state.currentPlayingUrl && !this.state.userStartedPlaying) {
                this.state.userStartedPlaying = true;
                // 新增：在视频开始播放后延迟预加载其他分集
                this.schedulePreloadOtherEpisodes();
            }
        });

        // 视频暂停事件 - 额外保存历史记录
        this.elements.video.addEventListener('pause', () => {
            if (this.state.currentPlayingUrl && this.elements.video.currentTime >= this.config.minSaveTime) {
                this.saveHistory(this.state.currentPlayingUrl, this.elements.video.currentTime);
            }
        });

        // 页面关闭前保存历史记录
        window.addEventListener('beforeunload', () => {
            if (this.state.currentPlayingUrl && this.elements.video && this.elements.video.currentTime >= this.config.minSaveTime) {
                this.saveHistory(this.state.currentPlayingUrl, this.elements.video.currentTime);
            }
        });

        // 播放结束后清理历史
        this.elements.video.addEventListener('ended', () => {
            if (this.state.currentPlayingUrl) {
                this.removeHistory(this.state.currentPlayingUrl);
            }
            
            // 添加自动播放下一集功能
            this.handleAutoPlayNext();
        });

        // 绑定辅助功能按钮事件
        this.bindHelperButtons();
        
        // 新增：绑定分集切换事件
        this.bindEpisodeSwitchEvents();
    }

    /**
     * 绑定辅助功能按钮事件
     */
    bindHelperButtons() {
        // 下载视频按钮事件
        const downloadBtn = document.getElementById('downloadVideoBtn');
        if (downloadBtn) {
            downloadBtn.addEventListener('click', () => this.downloadVideo());
        }
        
        // 关闭下载弹窗按钮事件
        const closeDownloadModalBtn = document.getElementById('closeDownloadModal');
        if (closeDownloadModalBtn) {
            closeDownloadModalBtn.addEventListener('click', () => this.closeDownloadModal());
        }
    }

    /**
     * 新增：绑定分集切换事件
     */
    bindEpisodeSwitchEvents() {
        // 为所有分集项添加点击事件监听
        document.addEventListener('DOMContentLoaded', () => {
            // 获取当前播放的剧集ID
            const activeEpisode = document.querySelector('.episode-item.bg-primary');
            if (activeEpisode) {
                this.currentEpisodeId = activeEpisode.getAttribute('data-episode-id');
            }

            // 监听分集项的点击事件
            const episodeItems = document.querySelectorAll('.episode-item');
            episodeItems.forEach(item => {
                item.addEventListener('click', (e) => {
                    // 更新当前剧集ID
                    this.currentEpisodeId = e.currentTarget.getAttribute('data-episode-id');
                });
            });
        });
    }

    /**
     * 新增：安排预加载其他分集
     */
    schedulePreloadOtherEpisodes() {
        // 清除之前的超时任务
        if (this.preloadTimeoutId) {
            clearTimeout(this.preloadTimeoutId);
        }

        // 延迟执行预加载，确保当前视频加载优先级更高
        this.preloadTimeoutId = setTimeout(() => {
            this.preloadOtherEpisodes();
        }, this.config.preloadDelay);
    }

    /**
     * 新增：预加载其他分集
     */
    preloadOtherEpisodes() {
        // 检查是否存在其他分集
        const episodeItems = document.querySelectorAll('.episode-item');
        if (!episodeItems || episodeItems.length < 2) {
            return; // 只有一集，无需预加载
        }

        // 清空预加载队列
        this.preloadQueue = [];

        // 查找当前剧集和其他分集
        episodeItems.forEach(item => {
            const episodeId = item.getAttribute('data-episode-id');
            const movieId = item.getAttribute('data-movie-id');
            
            // 跳过当前正在播放的剧集
            if (episodeId === this.currentEpisodeId) {
                return;
            }

            // 跳过已经预加载过的剧集
            const cacheKey = `${movieId}_${episodeId}`;
            if (this.preloadedEpisodes.has(cacheKey)) {
                return;
            }

            // 添加到预加载队列
            this.preloadQueue.push({ episodeId, movieId, cacheKey });
        });

        // 开始执行预加载队列
        this.processPreloadQueue();
    }

    /**
     * 处理预加载队列
     */
    processPreloadQueue() {
        // 如果队列已空或已有预加载任务在执行，则停止
        if (this.isPreloading || this.preloadQueue.length === 0) {
            return;
        }

        // 标记为正在预加载
        this.isPreloading = true;

        // 获取队列中的第一个任务
        const preloadTask = this.preloadQueue.shift();
        const { episodeId, movieId, cacheKey } = preloadTask;

        // 请求分集的播放地址
        this.getEpisodePlayUrl(movieId, episodeId)
            .then(playUrl => {
                if (playUrl) {
                    // 检查是否是B站链接格式
                    if (playUrl.includes('bilibili/index.php?url=')) {
                        console.log(`开始预加载B站分集 ${episodeId}: ${playUrl}`);
                        // 预加载1B数据
                        return this.preloadEpisodeData(playUrl, cacheKey);
                    } else {
                        // 对于非B站链接，直接进入finally
                        this.isPreloading = false;
                        setTimeout(() => this.processPreloadQueue(), 1000);
                        return Promise.resolve(null); // 特殊标记，表示跳过处理
                    }
                }
                return Promise.resolve(null); // 特殊标记，表示跳过处理
            })
            .then(success => {
                // 只有当success不是null时才处理
                if (success !== null) {
                    if (success) {
                        console.log(`分集 ${episodeId} 预加载成功`);
                        // 标记为已预加载
                        this.preloadedEpisodes.add(cacheKey);
                    } else {
                        console.log(`分集 ${episodeId} 预加载失败或已在缓存中`);
                    }
                }
            })
            .catch(error => {
                console.error(`分集 ${episodeId} 预加载错误:`, error);
            })
            .finally(() => {
                // 只有当success为null时不执行这里的逻辑，因为已经在then中处理了
                if (this.isPreloading) {
                    // 标记为不在预加载
                    this.isPreloading = false;
                    // 继续处理队列中的下一个任务
                    setTimeout(() => this.processPreloadQueue(), 1000); // 添加小延迟，避免过度并发
                }
            });
    }

    /**
     * 新增：获取分集的播放地址
     */
    getEpisodePlayUrl(movieId, episodeId) {
        return new Promise((resolve) => {
            // 首先尝试从DOM中获取（如果已经存在）
            const episodeItem = document.querySelector(`.episode-item[data-episode-id="${episodeId}"]`);
            if (episodeItem && episodeItem.dataset.playUrl) {
                resolve(episodeItem.dataset.playUrl);
                return;
            }

            // 否则通过AJAX请求获取
            fetch(`episode_load_ajax.php?movie_id=${movieId}&episode_id=${episodeId}`)
                .then(response => response.json())
                .then(data => {
                    if (data && data.play_url) {
                        // 缓存到DOM中
                        if (episodeItem) {
                            episodeItem.dataset.playUrl = data.play_url;
                        }
                        resolve(data.play_url);
                    } else {
                        resolve(null);
                    }
                })
                .catch(error => {
                    console.error('获取分集播放地址失败:', error);
                    resolve(null);
                });
        });
    }

    /**
     * 新增：预加载分集数据（1B）
     */
    preloadEpisodeData(url, cacheKey) {
        return new Promise((resolve) => {
            // 直接使用GET请求预加载1B数据，不使用HEAD请求避免错误提示
            fetch(url, {
                method: 'GET',
                mode: 'cors',
                headers: { 'Range': `bytes=0-${this.config.preloadSize - 1}` }, // 预加载1B数据
                cache: 'force-cache',
                priority: 'low'
            }).then(response => {
                if (response.ok || response.status === 206) { // 206 Partial Content
                    // 读取数据以确保它被缓存
                    response.arrayBuffer().then(() => {
                        resolve(true);
                    }).catch(() => {
                        resolve(false);
                    });
                } else {
                    resolve(false);
                }
            }).catch(() => {
                resolve(false);
            });
        });
    }

    /**
     * 新增：检查分集是否已预加载
     */
    isEpisodePreloaded(movieId, episodeId) {
        const cacheKey = `${movieId}_${episodeId}`;
        return this.preloadedEpisodes.has(cacheKey);
    }

    /**
     * 播放视频核心功能（优化版）
     */
    async playVideo(url) {
        if (!url) return;
        
        console.log('开始播放视频:', url);
        
        // 取消之前的预加载
        this.state.preloadingUrl = null;

        // 重置 UI / 状态
        this.state.currentPlayingUrl = url;
        this.state.currentVideoType = null;
        this.state.lastHistorySaveTime = 0; // 重置上次保存时间

        // 清除旧的 HLS 实例和 video src
        if (this.state.hls) {
            this.state.hls.destroy();
            this.state.hls = null;
        }
        this.elements.video.src = '';
        this.elements.video.pause();

        try {
            // 预先检查历史记录，为用户提供即时反馈
            const history = this.getHistory(url);
            if (history && history.time >= this.config.minSaveTime) {
                console.log('检测到历史记录，将在视频加载后恢复播放位置');
            }
            
            // 检测资源类型
            const type = await this.detectContentType(url);

            if (type === 'm3u8') {
                // 明确是 m3u8 → 使用 hls.js
                this.state.currentVideoType = 'm3u8';
                await this.playWithHLS(url);
            } else if (type === 'binary') {
                // 明确是普通视频文件 → 直接用 <video> 播放
                this.state.currentVideoType = 'binary';
                await this.playWithVideoTag(url);
            } else {
                // 未能判断 → 先尝试普通播放，若失败再回退到 hls.js
                try {
                    this.state.currentVideoType = 'binary';
                    await this.playWithVideoTag(url);
                } catch (videoErr) {
                    console.log('直接播放失败，尝试使用 hls.js:', videoErr);
                    this.state.currentVideoType = 'm3u8';
                    await this.playWithHLS(url);
                }
            }

        } catch (err) {
            console.error('视频播放错误:', err);
            this.showMessage('视频播放失败，请稍后再试或联系管理员');
        }
    }

    /**
     * 添加预加载视频功能
     */
    preloadVideo(url) {
        if (!url || this.state.preloadingUrl === url) return;
        
        this.state.preloadingUrl = url;
        
        // 使用低优先级预加载，不阻塞主要资源
        if ('requestIdleCallback' in window) {
            window.requestIdleCallback(() => {
                this._startPreloading(url);
            });
        } else {
            setTimeout(() => {
                this._startPreloading(url);
            }, 1000);
        }
    }

    /**
     * 内部预加载实现
     */
    _startPreloading(url) {
        // 仅预加载少量数据，用于后续快速启动
        fetch(url, { 
            method: 'GET',
            mode: 'cors',
            headers: { 'Range': 'bytes=0-524288' }, // 预加载512KB
            cache: 'force-cache',
            priority: 'low'
        }).catch(err => {
            console.log('预加载失败:', err);
        });
    }

    /**
     * 视频源类型检测（优化版）
     */
    async detectContentType(url) {
        // 添加URL类型缓存，避免重复检测
        if (this.contentTypeCache && this.contentTypeCache[url]) {
            return this.contentTypeCache[url];
        }

        try {
            // 优先通过URL扩展名快速判断
            if (url.toLowerCase().endsWith('.m3u8')) {
                this.contentTypeCache[url] = 'm3u8';
                return 'm3u8';
            }
            
            // 直接使用GET请求+Range，减少请求次数
            const response = await fetch(url, {
                method: 'GET',
                mode: 'cors',
                headers: { 'Range': 'bytes=0-1023' }, // 适当增加首字节获取量
                cache: 'force-cache'
            });
            
            if (!response.ok) {
                this.contentTypeCache[url] = null;
                return null;
            }

            const contentType = response.headers.get('Content-Type') || '';
            let type;
            
            if (contentType.includes('application/x-mpegURL') ||
                contentType.includes('application/vnd.apple.mpegurl')) {
                type = 'm3u8';
            } else if (contentType.startsWith('video/') || url.match(/\.(mp4|webm|mov|avi|mkv)$/i)) {
                type = 'binary';
            } else {
                type = null; // 未能判断
            }
            
            // 缓存结果
            this.contentTypeCache[url] = type;
            return type;
        } catch (e) {
            console.log('检测内容类型出错，将尝试自动播放:', e);
            // 缓存失败结果
            this.contentTypeCache[url] = null;
            return null;
        }
    }

    /**
     * 使用 hls.js 播放 m3u8（优化版）
     */
    playWithHLS(url) {
        return new Promise((resolve, reject) => {
            if (!window.Hls || !window.Hls.isSupported()) {
                return reject(new Error('您的浏览器不支持 HLS 播放'));
            }
            
            // 增强HLS配置，设置最大缓存
            this.state.hls = new window.Hls({
                enableWorker: true,
                maxBufferLength: 120, // 增加到120秒（原为15秒）
                maxBufferSize: 100 * 1024 * 1024, // 增加到100MB（原为10MB）
                backBufferLength: 60, // 增加到60秒
                maxBufferHole: 0.1, // 减少到0.1秒以更积极地填充缓冲
                startLevel: -1,
                autoStartLoad: true,
                lowBufferWatchdogPeriod: 1,
                highBufferWatchdogPeriod: 5, // 增加高缓冲监控周期
                nudgeOffset: 0.1,
                nudgeMaxRetry: 5, // 增加重试次数
                maxFragLookUpTolerance: 0.25
            });
            
            // 添加加载进度事件监听
            this.state.hls.on(window.Hls.Events.BUFFER_CREATED, () => {
                console.log('HLS: 缓冲区已创建');
            });
            
            this.state.hls.on(window.Hls.Events.BUFFER_APPENDING, (event, data) => {
                // 可以在这里添加缓冲进度显示
            });
            
            this.state.hls.on(window.Hls.Events.LEVEL_SWITCHING, (event, data) => {
                console.log(`HLS: 切换到质量等级 ${data.level}`);
            });
            
            // 添加MEDIA_ATTACHED事件，在视频元素绑定后检查历史记录
            this.state.hls.on(window.Hls.Events.MEDIA_ATTACHED, () => {
                console.log('HLS: 媒体已附加');
                const history = this.getHistory(url);
                if (history && history.time >= this.config.minSaveTime) {
                    console.log('HLS: 检测到历史记录，等待视频就绪后恢复播放位置');
                    // 延迟设置播放位置，确保视频已经准备好
                    this.elements.video.addEventListener('loadedmetadata', () => {
                        if (history && history.time >= this.config.minSaveTime) {
                            this.elements.video.currentTime = history.time;
                            console.log('HLS: 已自动恢复到历史观看时间:', this.formatTime(history.time));
                        }
                    }, { once: true });
                }
            });
            
            this.state.hls.loadSource(url);
            this.state.hls.attachMedia(this.elements.video);

            // 成功解析 Manifest 时即算成功
            this.state.hls.on(window.Hls.Events.MANIFEST_PARSED, resolve);

            // 增强错误处理和恢复机制
            this.state.hls.on(window.Hls.Events.ERROR, (event, data) => {
                console.error('HLS Error:', data);
                if (data.fatal) {
                    // 对不同错误做恢复尝试
                    switch (data.type) {
                        case window.Hls.ErrorTypes.NETWORK_ERROR:
                            console.log('HLS: 网络错误，尝试重新加载...');
                            // 增加重试延迟，避免频繁重试
                            setTimeout(() => this.state.hls.startLoad(), 1000);
                            break;
                        case window.Hls.ErrorTypes.MEDIA_ERROR:
                            console.log('HLS: 媒体错误，尝试恢复...');
                            this.state.hls.recoverMediaError();
                            break;
                        default:
                            reject(new Error('播放失败: ' + data.details));
                            break;
                    }
                } else {
                    // 处理非致命错误，如网络抖动
                    if (data.type === window.Hls.ErrorTypes.NETWORK_ERROR && data.details === 'manifestLoadError') {
                        console.log('HLS: 清单加载错误，尝试备用方案...');
                    }
                }
            });
        });
    }

    /**
     * 直接使用 <video> 播放二进制视频
     */
    playWithVideoTag(url) {
        return new Promise((resolve, reject) => {
            // 绑定成功/失败事件
            const onError = () => {
                this.elements.video.removeEventListener('error', onError);
                this.elements.video.removeEventListener('loadedmetadata', onLoad);
                reject(new Error('无法直接播放此视频'));
            };
            
            const onLoad = () => {
                this.elements.video.removeEventListener('error', onError);
                this.elements.video.removeEventListener('loadedmetadata', onLoad);
                // 同步速率
                this.elements.video.playbackRate = this.config.playbackRates[this.state.currentRateIndex];
                
                // 检查并应用历史记录时间
                const history = this.getHistory(url);
                if (history && history.time >= this.config.minSaveTime) {
                    this.elements.video.currentTime = history.time;
                    console.log('自动恢复到历史观看时间:', this.formatTime(history.time));
                    // 尝试自动播放
                    this.elements.video.play().catch(err => {
                        console.log('自动播放被浏览器阻止，等待用户交互:', err);
                        // 添加用户交互提示
                        this.showMessage('点击视频区域从上次观看位置继续');
                    });
                }
                
                resolve();
            };
            
            this.elements.video.addEventListener('error', onError);
            this.elements.video.addEventListener('loadedmetadata', onLoad);
            this.elements.video.src = url;
            this.elements.video.load();
        });
    }

    /**
     * 处理自动恢复播放
     */
    applyHistoryAndAutoPlay(url, time) {
        if (!this.elements.video || isNaN(time) || time < this.config.minSaveTime) {
            console.log('应用历史记录条件不满足:', !this.elements.video, isNaN(time), time < this.config.minSaveTime);
            return false;
        }
        
        try {
            console.log('尝试应用历史记录，URL:', url, '时间:', time);
            
            // 设置播放时间
            this.elements.video.currentTime = time;
            
            // 添加额外的延迟，确保currentTime设置生效
            setTimeout(() => {
                // 尝试自动播放
                this.elements.video.play().then(() => {
                    console.log('成功自动从历史记录恢复播放，时间:', this.formatTime(time));
                    // 标记用户已开始播放
                    this.state.userStartedPlaying = true;
                }).catch(err => {
                    console.log('自动播放被浏览器阻止，等待用户交互:', err);
                    // 添加用户交互提示
                    this.showMessage('点击视频区域从上次观看位置继续');
                });
            }, 100);
            
            return true;
        } catch (e) {
            console.error('应用历史记录失败:', e);
            return false;
        }
    }

    /**
     * 统一的消息显示方法
     */
    showMessage(message, duration = 3000) {
        // 移除已有的消息
        const existingMessage = document.querySelector('.shortcut-message');
        if (existingMessage) {
            existingMessage.remove();
        }
        
        // 创建消息元素
        const messageElement = document.createElement('div');
        messageElement.className = 'shortcut-message ajax-message';
        messageElement.style.cssText = `
            position: fixed;
            top: 20px;
            right: 20px;
            padding: 15px 20px;
            border-radius: 5px;
            z-index: 10000;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            max-width: 300px;
            background-color: #d1ecf1;
            color: #0c5460;
            font-size: 14px;
        `;
        
        messageElement.textContent = message;
        
        // 添加到页面
        document.body.appendChild(messageElement);
        
        // 3秒后自动消失
        setTimeout(() => {
            messageElement.style.transition = 'opacity 0.5s, transform 0.5s';
            messageElement.style.opacity = '0';
            messageElement.style.transform = 'translateX(100%)';
            
            setTimeout(() => {
                messageElement.remove();
            }, 500);
        }, duration);
    }
    
    /**
     * 处理键盘快捷键
     */
    handleKeyboardShortcuts(e) {
        // 当快捷键帮助浮窗显示时，ESC键关闭它
        const shortcutHelp = document.getElementById('shortcutHelp');
        if (shortcutHelp && !shortcutHelp.classList.contains('hidden') && e.key === 'Escape') {
            shortcutHelp.classList.add('hidden');
            return;
        }
        
        // 快捷键映射处理 - 简化为统一处理大小写
        const key = e.key.toLowerCase();
        
        const shortcutActions = {
            ' ': () => {
                this.togglePlayPause();
            },
            'enter': () => {
                this.toggleFullscreen();
                this.showMessage(document.fullscreenElement ? '已退出全屏' : '已进入全屏');
            },
            'f': () => {
                this.toggleFullscreenMode();
                this.showMessage(this.state.isFullscreenMode ? '已进入窗口全屏' : '已退出窗口全屏');
            },
            'p': () => {
                this.togglePictureInPicture();
                this.showMessage(document.pictureInPictureElement ? '已进入画中画' : '已退出画中画');
            },
            '[': () => {
                this.decreasePlaybackRate();
                this.showMessage(`播放速率: ${this.config.playbackRates[this.state.currentRateIndex].toFixed(1)}x`);
            },
            ']': () => {
                this.increasePlaybackRate();
                this.showMessage(`播放速率: ${this.config.playbackRates[this.state.currentRateIndex].toFixed(1)}x`);
            },
            'd': () => {
                this.toggleDownloadModal();
                this.showMessage(this.elements.downloadModal.style.display === 'flex' ? '已打开下载器' : '已关闭下载器');
            },
            'm': () => {
                this.toggleMute();
                this.showMessage(this.elements.video.muted ? '已静音' : '已取消静音');
            },
            'arrowup': () => {
                this.increaseVolume();
                this.showMessage(`音量: ${Math.round(this.elements.video.volume * 100)}%`);
            },
            'arrowdown': () => {
                this.decreaseVolume();
                this.showMessage(`音量: ${Math.round(this.elements.video.volume * 100)}%`);
            },
            'arrowleft': () => {
                this.seekBackward();
                this.showMessage('快退10秒');
            },
            'arrowright': () => {
                this.seekForward();
                this.showMessage('快进10秒');
            },
            's': () => {
                this.copyShareableLink();
                // copyShareableLink方法内部会调用showMessage
            },
            'h': () => {
                if (shortcutHelp) {
                    shortcutHelp.classList.toggle('hidden');
                    this.showMessage(shortcutHelp.classList.contains('hidden') ? '已隐藏快捷键帮助' : '已显示快捷键帮助');
                }
            },
            'n': () => {
                this.toggleAutoPlayNext();
            },
            'escape': () => {
                // ESC键的主要处理已经在前面的代码中
                this.showMessage('功能已关闭');
            },
            'pageup': () => {
                const currentEpisode = document.querySelector('.episode-item.bg-primary');
                if (currentEpisode) {
                    const prevEpisode = currentEpisode.previousElementSibling;
                    if (prevEpisode && prevEpisode.classList.contains('episode-item')) {
                        prevEpisode.click();
                        this.showMessage('切换到上一集');
                    } else {
                        this.showMessage('已经是第一集');
                    }
                }
            },
            'pagedown': () => {
                const currentEpisode = document.querySelector('.episode-item.bg-primary');
                if (currentEpisode) {
                    const nextEpisode = currentEpisode.nextElementSibling;
                    if (nextEpisode && nextEpisode.classList.contains('episode-item')) {
                        nextEpisode.click();
                        this.showMessage('切换到下一集');
                    } else {
                        this.showMessage('已经是最后一集');
                    }
                }
            }
        };
        
        // 执行对应的操作
        if (shortcutActions[key]) {
            shortcutActions[key]();
        }
    }

    /**
     * 切换下载弹窗显示状态
     */
    toggleDownloadModal() {
        if (this.elements.downloadModal.style.display === 'flex') {
            // 如果已打开，则关闭下载弹窗
            this.closeDownloadModal();
        } else {
            // 否则打开下载功能
            this.downloadVideo();
        }
    }

    /**
     * 下载功能
     */
    downloadVideo() {
        if (!this.state.currentPlayingUrl) return;

        if (this.state.currentVideoType === 'binary') {
            // 直接二进制视频 → 在新标签页打开，触发浏览器的"另存为"
            const urlWithTimestamp = this.state.currentPlayingUrl.includes('?')
                ? `${this.state.currentPlayingUrl}&download=${Date.now()}`
                : `${this.state.currentPlayingUrl}?download=${Date.now()}`;

            const newWindow = window.open(urlWithTimestamp, '_blank');

            // 若浏览器拦截弹窗，提示用户手动复制链接
            if (!newWindow) {
                this.showMessage('浏览器阻止了弹出窗口，请手动复制视频链接下载');
            }
        } else {
            // m3u8 下载采用 "下载弹窗 + iframe" 方式
            const encodedUrl = encodeURIComponent(this.state.currentPlayingUrl);
            const targetUrl = `download/?source=${encodedUrl}`;
            const decodedTargetUrl = decodeURIComponent(targetUrl);

            // 暂停视频播放
            const wasPlaying = !this.elements.video.paused;
            if (wasPlaying) {
                this.elements.video.pause();
            }

            this.elements.downloadFrameContainer.innerHTML = ''; // 清空旧的 iframe

            // 创建加载动画（防止 iframe 载入慢时空白）
            const loader = document.createElement('div');
            loader.className = 'download-loader';
            loader.innerHTML = `
                <div class="download-loader-spinner"></div>
                <div class="download-loader-text">加载下载器...</div>
            `;
            this.elements.downloadFrameContainer.appendChild(loader);

            // 注入 iframe
            const iframe = document.createElement('iframe');
            iframe.src = decodedTargetUrl;
            iframe.style.width = '100%';
            iframe.style.height = '100%';
            iframe.style.border = 'none';
            iframe.onload = () => loader.remove();
            this.elements.downloadFrameContainer.appendChild(iframe);

            this.elements.downloadModal.style.display = 'flex';
            document.body.style.overflow = 'hidden'; // 禁止背景滚动
            this.elements.downloadModal.dataset.wasPlaying = wasPlaying; // 记住是否需要恢复播放
        }
    }

    /**
     * 关闭下载弹窗并在必要时恢复播放
     */
    closeDownloadModal() {
        const wasPlaying = this.elements.downloadModal.dataset.wasPlaying === 'true';
        this.elements.downloadModal.style.display = 'none';
        document.body.style.overflow = 'auto';
        if (wasPlaying && this.elements.video && this.elements.historyPrompt.classList.contains('hidden')) {
            try {
                this.elements.video.play();
            } catch (e) {
                console.log('恢复播放失败:', e);
            }
        }
    }

    /**
     * 播放控制功能
     */
    togglePlayPause() {
        if (this.elements.video.paused) {
            this.elements.video.play();
            this.hideHistoryPrompt(); // 开始播放时自动关闭历史提示
            this.showMessage('继续播放');
        } else {
            this.elements.video.pause();
            this.showMessage('已暂停');
        }
    }

    seekForward() {
        this.elements.video.currentTime += 10;
    }

    seekBackward() {
        this.elements.video.currentTime -= 10;
    }

    increaseVolume() {
        this.elements.video.volume = Math.min(1, this.elements.video.volume + 0.1);
    }

    decreaseVolume() {
        this.elements.video.volume = Math.max(0, this.elements.video.volume - 0.1);
    }

    toggleMute() {
        this.elements.video.muted = !this.elements.video.muted;
    }

    /**
     * 自定义全屏模式
     */
    toggleFullscreenMode() {
        this.state.isFullscreenMode = !this.state.isFullscreenMode;
        
        if (this.state.isFullscreenMode) {
            this.enterFullscreenMode();
        } else {
            this.exitFullscreenMode();
        }
    }

    /**
     * 进入自定义全屏模式
     */
    enterFullscreenMode() {
        // 保存原始样式和父元素，以便退出时恢复
        this.state.originalVideoStyle = {
            position: this.elements.video.style.position,
            top: this.elements.video.style.top,
            left: this.elements.video.style.left,
            width: this.elements.video.style.width,
            height: this.elements.video.style.height,
            zIndex: this.elements.video.style.zIndex
        };
        this.state.originalParent = this.elements.video.parentNode;
        this.state.originalNextSibling = this.elements.video.nextSibling;

        // 移动视频到全屏容器
        this.elements.fullscreenContainer.appendChild(this.elements.video);
        
        // 设置视频标签为全屏样式
        this.elements.video.style.position = 'absolute';
        this.elements.video.style.top = '0';
        this.elements.video.style.left = '0';
        this.elements.video.style.width = '100%';
        this.elements.video.style.height = '100%';
        this.elements.video.style.objectFit = 'contain';
        this.elements.video.style.background = 'black';
        
        // 显示全屏容器
        this.elements.fullscreenContainer.style.display = 'block';
        
        // 处理所有控件元素
        this.adjustControlsForFullscreen();
        
        // 禁止页面滚动并保存滚动位置
        document.body.style.overflow = 'hidden';
        this.state.scrollTop = window.pageYOffset || document.documentElement.scrollTop;
    }

    /**
     * 退出自定义全屏模式
     */
    exitFullscreenMode() {
        // 恢复视频原始位置和样式
        if (this.state.originalParent && this.state.originalNextSibling) {
            this.state.originalParent.insertBefore(this.elements.video, this.state.originalNextSibling);
        } else if (this.state.originalParent) {
            this.state.originalParent.appendChild(this.elements.video);
        }
        
        // 恢复视频原始样式
        if (this.state.originalVideoStyle) {
            this.elements.video.style.position = this.state.originalVideoStyle.position;
            this.elements.video.style.top = this.state.originalVideoStyle.top;
            this.elements.video.style.left = this.state.originalVideoStyle.left;
            this.elements.video.style.width = this.state.originalVideoStyle.width;
            this.elements.video.style.height = this.state.originalVideoStyle.height;
            this.elements.video.style.zIndex = this.state.originalVideoStyle.zIndex;
            this.elements.video.style.objectFit = '';
            this.elements.video.style.background = '';
        }
        
        // 隐藏全屏容器
        this.elements.fullscreenContainer.style.display = 'none';
        
        // 恢复控件样式
        this.restoreControlsFromFullscreen();
        
        // 恢复页面滚动和滚动位置
        document.body.style.overflow = '';
        window.scrollTo(0, this.state.scrollTop || 0);
    }

    /**
     * 调整控件以适应全屏
     */
    adjustControlsForFullscreen() {
        const controls = document.querySelectorAll('.rate-indicator, .exit-fullscreen-hint');
        controls.forEach(control => {
            this.state.originalControlStyles = this.state.originalControlStyles || {};
            if (!this.state.originalControlStyles[control.className]) {
                this.state.originalControlStyles[control.className] = {
                    position: control.style.position,
                    zIndex: control.style.zIndex,
                    fontSize: control.style.fontSize
                };
            }
            control.style.position = 'fixed';
            control.style.zIndex = this.config.controlZIndex;
            // 适配全屏的控件大小
            if (control.className.includes('rate-indicator')) {
                control.style.fontSize = '16px';
            }
        });
    }

    /**
     * 从全屏恢复控件样式
     */
    restoreControlsFromFullscreen() {
        const controls = document.querySelectorAll('.rate-indicator, .exit-fullscreen-hint');
        controls.forEach(control => {
            if (this.state.originalControlStyles && this.state.originalControlStyles[control.className]) {
                control.style.position = this.state.originalControlStyles[control.className].position;
                control.style.zIndex = this.state.originalControlStyles[control.className].zIndex;
                control.style.fontSize = this.state.originalControlStyles[control.className].fontSize;
            }
        });
    }

    /**
     * 原生全屏（Browser Fullscreen API）
     */
    toggleFullscreen() {
        if (!document.fullscreenElement) {
            // 进入全屏
            if (this.elements.video.requestFullscreen) this.elements.video.requestFullscreen();
            else if (this.elements.video.mozRequestFullScreen) this.elements.video.mozRequestFullScreen();
            else if (this.elements.video.webkitRequestFullscreen) this.elements.video.webkitRequestFullscreen();
            else if (this.elements.video.msRequestFullscreen) this.elements.video.msRequestFullscreen();
        } else {
            // 退出全屏
            if (document.exitFullscreen) document.exitFullscreen();
            else if (document.mozCancelFullScreen) document.mozCancelFullScreen();
            else if (document.webkitExitFullscreen) document.webkitExitFullscreen();
            else if (document.msExitFullscreen) document.msExitFullscreen();
        }
    }

    /**
     * 播放速率控制
     */
    increasePlaybackRate() {
        if (this.state.currentRateIndex < this.config.playbackRates.length - 1) {
            this.state.currentRateIndex++;
            this.applyPlaybackRate();
        }
    }

    decreasePlaybackRate() {
        if (this.state.currentRateIndex > 0) {
            this.state.currentRateIndex--;
            this.applyPlaybackRate();
        }
    }

    applyPlaybackRate() {
        if (this.elements.video) {
            this.elements.video.playbackRate = this.config.playbackRates[this.state.currentRateIndex];
        }
    }

    /**
     * 画中画（Picture‑in‑Picture）
     */
    async togglePictureInPicture() {
        try {
            if (document.pictureInPictureElement) {
                await document.exitPictureInPicture();
            } else if (document.pictureInPictureEnabled && this.elements.video) {
                await this.elements.video.requestPictureInPicture();
            }
        } catch (err) {
            console.error('画中画错误:', err);
            this.showMessage('画中画模式不可用');
        }
    }

    /**
     * 分享链接
     */
    copyShareableLink() {
        if (!this.state.currentPlayingUrl) return;
        
        const baseUrl = window.location.href.split('?')[0];
        const movieId = new URLSearchParams(window.location.search).get('id');
        const episodeId = new URLSearchParams(window.location.search).get('episode_id');
        
        let shareableUrl = `${baseUrl}?id=${movieId}`;
        if (episodeId) {
            shareableUrl += `&episode_id=${episodeId}`;
        }
        
        this.copyToClipboard(shareableUrl);
        this.showMessage('链接已复制到剪贴板 (快捷键: S)');
    }

    /**
     * 历史记录功能 - 保存
     */
    saveHistory(url, time) {
        if (!url || time < this.config.minSaveTime) return; // 过滤太小的时间点
        
        // 添加节流，避免过于频繁的保存
        const now = Date.now();
        if (now - this.state.lastHistorySaveTime < this.config.historySaveInterval * 1000) {
            return;
        }
        
        try {
            // 检查localStorage是否可用
            if (typeof Storage === 'undefined') {
                console.warn('浏览器不支持localStorage，无法保存历史记录');
                return;
            }
            
            // 检查存储容量
            const history = JSON.parse(localStorage.getItem('movieHistory') || '{}');
            history[url] = { time, timestamp: now, title: document.title };
            
            // 尝试保存数据，如果失败则清理旧数据并重试
            try {
                localStorage.setItem('movieHistory', JSON.stringify(history));
                this.state.lastHistorySaveTime = now;
            } catch (e) {
                if (e.name === 'QuotaExceededError') {
                    console.warn('localStorage空间不足，尝试清理旧数据...');
                    this.cleanupOldHistory();
                    // 再次尝试保存
                    localStorage.setItem('movieHistory', JSON.stringify(history));
                    this.state.lastHistorySaveTime = now;
                } else {
                    throw e;
                }
            }
        } catch (e) {
            console.error('保存历史记录失败:', e);
        }
    }

    /**
     * 清理旧的历史记录数据
     */
    cleanupOldHistory() {
        try {
            const history = JSON.parse(localStorage.getItem('movieHistory') || '{}');
            const historyArray = Object.entries(history);
            
            // 按时间戳排序，保留最新的10条记录
            if (historyArray.length > 10) {
                historyArray.sort((a, b) => b[1].timestamp - a[1].timestamp);
                const newHistory = {};
                
                // 只保留最新的10条记录
                for (let i = 0; i < Math.min(10, historyArray.length); i++) {
                    newHistory[historyArray[i][0]] = historyArray[i][1];
                }
                
                localStorage.setItem('movieHistory', JSON.stringify(newHistory));
                console.log('已清理旧的历史记录数据');
            }
        } catch (e) {
            console.error('清理历史记录失败:', e);
        }
    }

    /**
     * 历史记录功能 - 获取
     */
    getHistory(url) {
        try {
            // 检查localStorage是否可用
            if (typeof Storage === 'undefined') {
                return null;
            }
            
            const history = JSON.parse(localStorage.getItem('movieHistory') || '{}');
            return history[url] || null;
        } catch (e) {
            console.error('获取历史记录失败:', e);
            return null;
        }
    }

    /**
     * 历史记录功能 - 删除
     */
    removeHistory(url) {
        try {
            // 检查localStorage是否可用
            if (typeof Storage === 'undefined') {
                return;
            }
            
            const history = JSON.parse(localStorage.getItem('movieHistory') || '{}');
            if (history[url]) {
                delete history[url];
                localStorage.setItem('movieHistory', JSON.stringify(history));
            }
        } catch (e) {
            console.error('删除历史记录失败:', e);
        }
    }

    /**
     * 显示历史记录提示
     */
    showHistoryPrompt(time) {
        if (!this.elements.historyPrompt || !this.elements.historyTimeSpan) {
            console.warn('历史记录提示窗口元素未找到');
            return;
        }

        // 检查时间有效性
        if (isNaN(time) || time < this.config.minSaveTime) {
            console.warn('无效的历史记录时间');
            return;
        }

        // 格式化并显示时间
        this.elements.historyTimeSpan.textContent = this.formatTime(time);

        // 重置样式，确保窗口能够正确显示
        this.elements.historyPrompt.style.transition = 'opacity 0.3s ease-in-out';
        this.elements.historyPrompt.style.opacity = '0';
        
        // 确保元素有正确的z-index和定位
        this.elements.historyPrompt.style.zIndex = '9999';
        this.elements.historyPrompt.style.position = 'fixed';
        this.elements.historyPrompt.style.pointerEvents = 'auto'; // 确保可点击

        // 确保提示窗口显示
        this.elements.historyPrompt.classList.remove('hidden');
        
        // 强制重排
        this.elements.historyPrompt.offsetHeight;
        
        // 使用更简单可靠的淡入动画
        setTimeout(() => {
            this.elements.historyPrompt.style.opacity = '1';
            
            // 3秒后如果用户没有操作，自动应用历史记录并开始播放
            setTimeout(() => {
                if (this.elements.historyPrompt && !this.elements.historyPrompt.classList.contains('hidden')) {
                    const url = this.state.currentPlayingUrl;
                    if (url) {
                        this.applyHistoryAndAutoPlay(url, time);
                    }
                }
            }, 3000);
        }, 10);
    }

    /**
     * 隐藏历史记录提示
     */
    hideHistoryPrompt() {
        if (!this.elements.historyPrompt) {
            return;
        }
        
        // 添加淡出动画
        this.elements.historyPrompt.style.transition = 'opacity 0.3s ease-in-out';
        this.elements.historyPrompt.style.opacity = '0';
        
        // 动画完成后隐藏
        setTimeout(() => {
            this.elements.historyPrompt.classList.add('hidden');
            // 仅在用户未明确操作且没有历史记录时，才考虑从头自动播放
            if (!this.state.userStartedPlaying && this.elements.video && this.elements.video.currentTime === 0) {
                this.elements.video.play().catch(err => {
                    console.log('自动播放被浏览器阻止:', err);
                });
            }
        }, 300);
    }

    /**
     * 切换自动播放下一集的状态
     */
    toggleAutoPlayNext() {
        const autoPlayNextToggle = document.getElementById('autoPlayNextToggle');
        if (autoPlayNextToggle) {
            const checkbox = autoPlayNextToggle.querySelector('input[type="checkbox"]');
            if (checkbox) {
                checkbox.checked = !checkbox.checked;
                this.showMessage(checkbox.checked ? '已开启自动下一集' : '已关闭自动下一集');
            }
        }
    }

    /**
     * 处理视频播放结束后的自动播放下一集逻辑
     */
    handleAutoPlayNext() {
        // 检查自动播放开关是否开启
        const autoPlayNextToggle = document.getElementById('autoPlayNextToggle');
        const isAutoPlayEnabled = autoPlayNextToggle && autoPlayNextToggle.querySelector('input').checked;
        
        if (!isAutoPlayEnabled) {
            return;
        }
        
        // 查找当前选中的剧集和下一集
        const currentEpisode = document.querySelector('.episode-item.bg-primary');
        if (currentEpisode) {
            const nextEpisode = currentEpisode.nextElementSibling;
            if (nextEpisode && nextEpisode.classList.contains('episode-item')) {
                // 延迟一小段时间后播放下一集，避免用户体验突兀
                setTimeout(() => {
                    this.showMessage('即将自动播放下一集...');
                    // 再次确认开关是否开启，防止用户在延迟期间关闭
                    const stillEnabled = autoPlayNextToggle && autoPlayNextToggle.querySelector('input').checked;
                    if (stillEnabled) {
                        nextEpisode.click();
                    }
                }); 
            }
        }
    }

    /**
     * 专门处理分集切换的方法
     * 集中管理清除旧状态、加载新资源、应用历史记录等所有相关逻辑
     * @param {string} newEpisodeUrl - 新分集的播放URL
     * @param {number} episodeNum - 分集编号（可选）
     * @param {string} episodeTitle - 分集标题（可选）
     * @param {string} movieTitle - 电影标题（可选）
     */
    async switchEpisode(newEpisodeUrl, episodeNum = null, episodeTitle = null, movieTitle = null) {
        try {
            // 1. 保存当前播放进度（如果有正在播放的视频）
            if (this.state.currentPlayingUrl && this.elements.video && 
                this.elements.video.currentTime >= this.config.minSaveTime) {
                this.saveHistory(this.state.currentPlayingUrl, this.elements.video.currentTime);
                console.log('已保存当前剧集观看进度');
            }

            // 保存用户是否已经开始播放的状态
            const wasUserPlaying = this.state.userStartedPlaying;
            // 保存当前播放速率
            const savedRateIndex = this.state.currentRateIndex;

            // 2. 清除旧状态和资源
            this.clearEpisodeState();

            // 3. 加载新的分集视频
            await this.playVideo(newEpisodeUrl);

            // 恢复保存的播放速率
            this.state.currentRateIndex = savedRateIndex;
            this.applyPlaybackRate();

            // 4. 更新URL但不刷新页面（保持浏览历史连续性）
            this.updateUrlWithoutRefresh(episodeNum);

            // 5. 更新浏览器标题
            if (movieTitle && episodeNum) {
                let titleSuffix = '';
                if (episodeTitle) {
                    titleSuffix = ' - ' + episodeTitle;
                } else if (episodeNum) {
                    titleSuffix = ' - 第' + episodeNum + '集';
                }
                document.title = movieTitle + titleSuffix + ' - 爱影视';
            }

            // 检查是否有历史记录并且用户之前已经开始播放
            const history = this.getHistory(newEpisodeUrl);
            if (history && history.time >= this.config.minSaveTime && wasUserPlaying) {
                // 如果用户之前已经开始播放，则尝试自动从历史记录恢复播放
                console.log('尝试从历史记录自动恢复播放');
                this.applyHistoryAndAutoPlay(newEpisodeUrl, history.time);
            }

            console.log(`分集切换成功: ${episodeNum || '未知集数'}`);
        } catch (error) {
            console.error('分集切换失败:', error);
            // 显示错误消息
            this.showMessage('分集切换失败，请稍后再试');
        } finally {
            // 不再隐藏加载状态，因为我们已经移除了显示加载状态的代码
        }
    }

    /**
     * 清除剧集相关的状态和资源
     * 内部辅助方法
     */
    clearEpisodeState() {
        // 重置相关状态
        this.state.currentPlayingUrl = '';
        this.state.currentVideoType = null;
        this.state.lastHistorySaveTime = 0;
        this.state.userStartedPlaying = false;
        this.state.preloadingUrl = null;

        // 清除HLS实例和视频源
        if (this.state.hls) {
            this.state.hls.destroy();
            this.state.hls = null;
        }
        
        // 暂停当前视频
        if (this.elements.video) {
            this.elements.video.pause();
            this.elements.video.src = '';
        }

        // 隐藏相关UI元素
        this.hideHistoryPrompt();
        this.closeDownloadModal();
        
        console.log('已清除旧的剧集状态');
    }

    /**
     * 更新URL但不刷新页面
     * 内部辅助方法
     * @param {number} episodeNum - 分集编号（可选）
     */
    updateUrlWithoutRefresh(episodeNum = null) {
        try {
            const currentUrl = new URL(window.location.href);
            const movieId = currentUrl.searchParams.get('id');
            const currentEpisodeId = currentUrl.searchParams.get('episode_id');
            
            // 如果是通过episode_load_ajax.php加载，则需要更新URL中的episode_id
            if (movieId) {
                const newUrl = new URL(window.location.href);
                
                // 如果有新的剧集ID（可以从参数或DOM中获取）
                // 这里假设DOM中当前选中的episode-item包含data-episode-id
                const activeEpisodeItem = document.querySelector('.episode-item.bg-primary');
                if (activeEpisodeItem) {
                    const newEpisodeId = activeEpisodeItem.getAttribute('data-episode-id');
                    if (newEpisodeId && newEpisodeId !== currentEpisodeId) {
                        newUrl.searchParams.set('episode_id', newEpisodeId);
                    }
                }
                
                window.history.pushState({ path: newUrl.href }, '', newUrl.href);
            }
        } catch (error) {
            console.warn('更新URL失败:', error);
        }
    }

    /**
     * 工具函数
     */
    copyToClipboard(text) {
        const textarea = document.createElement('textarea');
        textarea.value = text;
        document.body.appendChild(textarea);
        textarea.select();
        document.execCommand('copy');
        document.body.removeChild(textarea);
    }

    /**
     * 格式化时间
     */
    formatTime(seconds) {
        const h = Math.floor(seconds / 3600);
        const m = Math.floor((seconds % 3600) / 60);
        const s = Math.floor(seconds % 60);
        return h > 0
            ? `${h}:${m.toString().padStart(2, '0')}:${s.toString().padStart(2, '0')}`
            : `${m}:${s.toString().padStart(2, '0')}`;
    }

    /**
     * 防抖函数
     */
    debounce(func, wait) {
        let timeout;
        return function executedFunction(...args) {
            const later = () => {
                clearTimeout(timeout);
                func(...args);
            };
            clearTimeout(timeout);
            timeout = setTimeout(later, wait);
        };
    }
}

// 创建播放器实例
const videoPlayer = new VideoPlayer();