<?php
// 搜索结果页面
$pageTitle = '搜索结果';
$activePage = 'search';

// 引入数据库配置
require_once 'config.php';

// 获取搜索关键词
$keyword = isset($_GET['keyword']) ? trim($_GET['keyword']) : '';

// 设置分页参数
$page = isset($_GET['page']) ? intval($_GET['page']) : 1;
$pageSize = 12; // 与分类页保持一致
$offset = ($page - 1) * $pageSize;

// 初始化查询结果
$movies = [];
$totalMovies = 0;
$totalPages = 0;

// 如果有搜索关键词，执行搜索
if (!empty($keyword)) {
    try {
        // 获取数据库连接
        $conn = getDbConnection();
        
        // 构建搜索查询，搜索多个字段
        $searchTerm = '%' . $keyword . '%';
        
        // 创建统一的状态过滤条件SQL片段
        $statusFilterSql = "INNER JOIN video_types vt ON m.type = vt.id 
            INNER JOIN movie_categories mc ON m.category_id = mc.id 
            WHERE m.status = 1 
            AND vt.status = 1 
            AND mc.status = 1";
        
        // 构建搜索条件SQL片段
        $searchConditionSql = "AND ( 
            m.title LIKE ? OR 
            m.subtitle LIKE ? OR 
            m.actors LIKE ? OR 
            m.director LIKE ? OR 
            m.description LIKE ?
        )";
        
        // 计算总记录数 - 使用统一的状态过滤逻辑
        $stmt = $conn->prepare("SELECT COUNT(m.id) FROM movies m $statusFilterSql $searchConditionSql");
        $stmt->execute([$searchTerm, $searchTerm, $searchTerm, $searchTerm, $searchTerm]);
        $totalMovies = $stmt->fetchColumn();
        
        // 计算总页数
        $totalPages = ceil($totalMovies / $pageSize);
        
        // 确保页码有效
        $page = max(1, min($page, $totalPages));
        $offset = ($page - 1) * $pageSize;
        
        // 查询当前页的电影 - 使用统一的状态过滤逻辑
        $stmt = $conn->prepare("SELECT m.* FROM movies m $statusFilterSql $searchConditionSql 
            ORDER BY m.create_time DESC
            LIMIT ?, ?
        ");
        $stmt->execute([$searchTerm, $searchTerm, $searchTerm, $searchTerm, $searchTerm, $offset, $pageSize]);
        $movies = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
    } catch (PDOException $e) {
        // 错误处理，实际项目中应该记录日志
        echo "数据库错误: " . $e->getMessage();
        exit;
    }
}

// 引入头部
include 'includes/header.php';
?>

<div class="container mx-auto px-4 py-8 max-w-7xl">
    <!-- 搜索结果头部 -->
    <h2 class="text-3xl font-bold mb-6">
        <?php if (!empty($keyword)): ?>
            "<?php echo htmlspecialchars($keyword); ?>" 的搜索结果
        <?php else: ?>
            搜索
        <?php endif; ?>
    </h2>
    
    <?php if (!empty($keyword)): ?>
        <p class="text-muted mb-6">找到 <?php echo number_format($totalMovies); ?> 个相关结果</p>
    <?php endif; ?>


    <?php if (!empty($keyword)): ?>
        <!-- 搜索结果列表 -->
        <div class="movie-list grid grid-cols-2 sm:grid-cols-3 md:grid-cols-4 lg:grid-cols-6 gap-4 mb-8">
            <?php if (empty($movies)): ?>
                <!-- 无搜索结果 -->
                <div class="no-results text-center py-12 col-span-full">
                    <div class="text-5xl mb-4">🔍</div>
                    <h3 class="text-xl font-bold mb-2">未找到相关内容</h3>
                    <p class="text-muted mb-6">尝试使用其他关键词搜索</p>
                    <a href="index.php" class="inline-flex items-center px-4 py-2 bg-primary hover:bg-primary/90 rounded-lg transition-colors">
                        <svg xmlns="http://www.w3.org/2000/svg" class="h-5 w-5 mr-2" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M10 19l-7-7m0 0l7-7m-7 7h18" />
                        </svg>
                        返回首页
                    </a>
                </div>
            <?php else: ?>
                <?php foreach ($movies as $movie): ?>
                    <div class="movie-item relative h-64 rounded-lg overflow-hidden shadow hover:-translate-y-2 hover:shadow-lg transition-all duration-300 bg-dark-light">
                        <a href="play.php?id=<?php echo $movie['id']; ?>" class="block h-full">
                            <img src="<?php echo $movie['cover_img'] ?: 'images/default_cover.jpg'; ?>" alt="<?php echo htmlspecialchars($movie['title']); ?>" class="absolute inset-0 w-full h-full object-cover transition-transform duration-500 hover:scale-110">
                            <!-- 新增：将热度放在右上角 -->
                            <div class="absolute top-2 right-2 bg-black/70 text-white text-xs px-2 py-1 rounded-full">
                                热度: <?php echo $movie['view_count']; ?>
                            </div>
                            <div class="absolute bottom-0 left-0 right-0 bg-gradient-to-t from-black/90 to-transparent p-3 text-white">
                                <h4 class="text-base font-medium line-clamp-1 mb-1"><?php echo htmlspecialchars($movie['title']); ?></h4>
                                <p class="text-xs text-gray-300 line-clamp-1 mb-1 opacity-80"><?php echo htmlspecialchars($movie['director']); ?> | <?php echo htmlspecialchars($movie['actors']); ?></p>
                                <!-- 修改：将年份和总集数放在同一行 -->
                                <div class="flex justify-between items-center">
                                    <p class="text-xs text-gray-400 opacity-80 m-0"><?php echo substr($movie['release_year'], 0, 4); ?></p>
                                    <?php if ($movie['total_episodes'] > 1): ?>
                                        <p class="text-xs text-gray-400 opacity-80 m-0">共<?php echo $movie['total_episodes']; ?>集</p>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </a>
                    </div>
                <?php endforeach; ?>
            <?php endif; ?>
        </div>
        
        <!-- 分页控件 -->
        <?php if ($totalPages > 1): ?>
            <div class="pagination flex justify-center mt-8 space-x-2">
                <a href="search.php?keyword=<?php echo urlencode($keyword); ?>&page=1" 
                   class="px-4 py-2 bg-secondary rounded-md text-primary hover:bg-primary hover:text-white transition-all <?php echo $page == 1 ? 'opacity-50 cursor-not-allowed' : ''; ?>" 
                   <?php echo $page == 1 ? 'aria-disabled="true"' : ''; ?>>首页</a>
                <a href="search.php?keyword=<?php echo urlencode($keyword); ?>&page=<?php echo $page - 1; ?>" 
                   class="px-4 py-2 bg-secondary rounded-md text-primary hover:bg-primary hover:text-white transition-all <?php echo $page == 1 ? 'opacity-50 cursor-not-allowed' : ''; ?>" 
                   <?php echo $page == 1 ? 'aria-disabled="true"' : ''; ?>>上一页</a>
                
                <?php for ($i = max(1, $page - 2); $i <= min($totalPages, $page + 2); $i++): ?>
                    <a href="search.php?keyword=<?php echo urlencode($keyword); ?>&page=<?php echo $i; ?>" 
                       class="px-4 py-2 rounded-md text-white transition-all <?php echo $page == $i ? 'bg-primary' : 'bg-secondary text-primary hover:bg-primary'; ?>"><?php echo $i; ?></a>
                <?php endfor; ?>
                
                <a href="search.php?keyword=<?php echo urlencode($keyword); ?>&page=<?php echo $page + 1; ?>" 
                   class="px-4 py-2 bg-secondary rounded-md text-primary hover:bg-primary hover:text-white transition-all <?php echo $page == $totalPages ? 'opacity-50 cursor-not-allowed' : ''; ?>" 
                   <?php echo $page == $totalPages ? 'aria-disabled="true"' : ''; ?>>下一页</a>
                <a href="search.php?keyword=<?php echo urlencode($keyword); ?>&page=<?php echo $totalPages; ?>" 
                   class="px-4 py-2 bg-secondary rounded-md text-primary hover:bg-primary hover:text-white transition-all <?php echo $page == $totalPages ? 'opacity-50 cursor-not-allowed' : ''; ?>" 
                   <?php echo $page == $totalPages ? 'aria-disabled="true"' : ''; ?>>末页</a>
            </div>
        <?php endif; ?>
    <?php endif; ?>
</div>

<?php
// 引入页脚
include 'includes/footer.php';
?>